/*
    AMVisC++Interface - C++ Interface for Body- and Pos-File Creation
                        Used by AMVis
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technical University of Munich

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef ARROW_H
#define ARROW_H

#include <cbody.h>

namespace AMVis {

//! Creates a arrow.
/*!
 * This class is used to create a arrow.
*/
class Arrow : public CBody {
public:
  //! Create a arrow.
  /*!
   * baseFilename is the basename of the body and pos file. instances is the
   * number of instances this body should create and if binPosFile is true the
   * pos file is created in binary mode or in ascii mode if false.
  */
  Arrow(const std::string &, int, bool);

  //! Set the size of head of the arrow.
  /*!
   * diameter is the diameter of the arrowhead and length the length of the arrowhead.
  */
  void setArrowHead(float diameter, float length);

  //! Set the diameter of the arrow body.
  void setDiameter(float diameter);

  //! Set the type of the arrow.
  /*!
   * type = 0: draw no head;
   * type = 1: draw only a to head;
   * type = 2: draw only a from head;
   * type = 3: draw both heads;
  */
  void setType(int type);

  //! Set scale factor for arrow lenght.
  void setScaleFactor(float scale);

  //! Write a arrow body file.
  virtual void writeBodyFile() const;

  //! Append a dataset to the pos file.
  /*!
   * Appends the current dataset (set with AMVis::Arrow::setTime,
   * AMVis::Arrow::setToPoint, AMVis::Arrow::setDirection
   * and AMVis::Arrow::setColor) to the pos file with index ind. ind runs from 0 to
   * instances-1.
  */
  virtual void appendDataset(int ind) const;

  //! Set the current simulation time.
  void setTime(float time);

  //! Set the to-point.
  /*!
   * x, y and z are the world coordinates of the to-point of the arrow.
  */
  void setToPoint(float x, float y, float z);

  //! Set the direction.
  /*!
   * dy, dy and dz is the direction of the arrow.
  */
  void setDirection(float dx, float dy, float dz);

  //! Set the color.
  /*!
   * color is the color of the rigid body at the time given by
   * AMVis::CRigidBody::setTime.
   * color runs from 0.0=blue over 0.25=cyan over 0.5=green over 0.75=magenta
   * to 1.0=red.
  */
  void setColor(float color);

protected:
  float length, diameter, diameterhead;
  int type;
  float scaleFactor;

  float t, x, y, z, dx, dy, dz, c;

  void writeToBodyFile(std::ofstream *) const;
};

}

#endif
