/*
   AMVisC++Interface - C++ Interface for Body- and Pos-File Creation
   Used by AMVis
   Copyright (C) 2006 Institute of Applied Mechanics,
   Technical University of Munich

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef COILSPRING_H
#define COILSPRING_H

#include <cbody.h>

namespace AMVis {

  //! Creates a spring.
  /*!
   * This class is used to create a spring, which can be moved and rotation in
   * space.
   */
  class CoilSpring : public CBody {
    public:
      //! Create a spring.
      /*!
       * baseFilename is the basename of the body and pos file. instances is the
       * number of instances this body should create and if binPosFile is true the
       * pos file is created in binary mode or in ascii mode if false.
       */
      CoilSpring(const std::string &, int, bool);

      //! Set the number of coils.
      void setNumberOfCoils(float);

      //! Set Radius of spring.
      void setRadius(float);

      //! Set Radius of cross section.
      void setRadiusCrossSection(float);

      //! Set the current simulation time.
      void setTime(float time);

      //! Set the to-point.
      /*!
       * x, y and z are the world coordinates of the to-point of the spring.
       */
      void setToPoint(float x, float y, float z);

      //! Set the from-point.
      /*!
       * x, y and z are the world coordinates of the from-point of the spring.
       */
      void setFromPoint(float x, float y, float z);

      //! Set the color.
      /*!
       * color is the color of the rigid body at the time given by
       * AMVis::CBody::setTime.
       * color runs from 0.0=blue over 0.25=cyan over 0.5=green over 0.75=magenta to 1.0=red.
       */
      void setColor(float color);

      //! Set scale factor for arrow lenght.
      void setScaleFactor(float scale);


      //! Write a body file.
      virtual void writeBodyFile() const;


      //! Append a dataset to the pos file.
      /*!
       * Appends the current dataset (set with AMVis::CoilSpring::setTime,
       * AMVis::CoilSpring::setFromPoint, AMVis::Arrow::setToPoint and AMVis::CoilSpring::setColor) 
       * to the pos file with index ind. ind runs from 0 to instances-1.
       */
      virtual void appendDataset(int ind) const;


    protected:
      float nCoils, rSpring, rCrossSection;
      float scaleFactor;

      float t, xFrom, yFrom, zFrom, xTo, yTo, zTo, c;

      void writeToBodyFile(std::ofstream *) const;
  };

}

#endif





