/*
    AMVisC++Interface - C++ Interface for Body- and Pos-File Creation
                        Used by AMVis
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technical University of Munich

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef CRIGIDBODY_H
#define CRIGIDBODY_H

#include <cbody.h>
#include <fstream>

namespace AMVis {

//! Abstract base class for all rigid bodies.
class CRigidBody : public CBody {

friend class CompoundPrimitiveBody;

public:
  //! Set the initial translation.
  /*!
   * x, y and z are the initial translation values of the rigid body.
  */
  void setInitialTranslation(float x, float y, float z);

  //! Set the initial rotation.
  /*!
   * alpha, beta and gamma are the initial rotation angles of the rigid body.
   * The angles are kardan angles and the measurement is radian.
  */
  void setInitialRotation(float alpha, float beta, float gamma);

  //! Set the scale factor of the body.
  void setScaleFactor(float scale);

  //! Set the current simulation time.
  void setTime(float time);

  //! Set the translation.
  /*!
   * x, y and z are the translation values of the rigid body at the time given
   * by AMVis::CRigidBody::setTime.
  */
  void setTranslation(float x, float y, float z);

  //! Set the rotation.
  /*!
   * alpha, beta and gamma are the rotation angles of the rigid body at the
   * time given by AMVis::CRigidBody::setTime. The angles are kardan angles and the
   * measurement is radian.
  */
  void setRotation(float alpha, float beta, float gamma);

  //! Set the color.
  /*!
   * color is the color of the rigid body at the time given by
   * AMVis::CRigidBody::setTime.
   * color runs from 0.0=blue over 0.25=cyan over 0.5=green over 0.75=magenta
   * to 1.0=red.
  */
  void setColor(float color);

  //! Append a dataset to the pos file.
  /*!
   * Appends the current dataset (set with AMVis::CRigidBody::setTime,
   * AMVis::CRigidBody::setTranslation, AMVis::CRigidBody::setRotation
   * and AMVis::CRigidBody::setColor) to the pos file with index ind. ind runs from 0 to
   * instances-1.
  */
  virtual void appendDataset(int ind) const;

protected:
  CRigidBody(const std::string &, int, bool);
  CRigidBody();
  float initialTranslation[3];
  float initialRotation[3];
  float scaleFactor;

  float t;
  float translation[3], rotation[3];
  float color;
 
  void writeToBodyFile(std::ofstream *) const;
};

}

#endif
