/*
   AMVisC++Interface - C++ Interface for Body- and Pos-File Creation
   Used by AMVis
   Copyright (C) 2006 Institute of Applied Mechanics,
   Technical University of Munich

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   */

#ifndef EXTRUSION_H
#define EXTRUSION_H

#include <crigidbody.h>
#include <vector>
#include <polygonpoint.h>

namespace AMVis {

  //! Creates a extrution body.
  /*!
   * Creates a extrusion body. The body is extruted in z-axis direction.
   * The base area is given by several user defined closed polygons. The
   * polygons are combined by a OpenGL winding rule.
   */
  class Extrusion : public CRigidBody {
    public:
      //! Create a extrusion body.
      /*!
       * baseFilename is the basename of the body and pos file. instances is the
       * number of instances this body should create and if binPosFile is true the
       * pos file is created in binary mode or in ascii mode if false.
       * If destroymem is true (default) then the destructor of this object will
       * delete all the memory allocated by a contour added with setContour.
       */
      Extrusion(const std::string &, int, bool, bool=true);

      //! Create a extrusion body.
      /*!
       * Use this constructor if you want to append this body to a
       * CompoundPrimitiveBody. This constructor dose not create a body of pos file
       * If destroymem is true (default) then the destructor of this object will
       * delete all the memory allocated by a contour added with setContour.
       */
      Extrusion(bool=true);

      //! Delete a extrustion body.
      ~Extrusion();

      //! Set winding rule.
      /*!
       * windingRule is the OpenGL winding rule, the polygon contours are combined
       * with. Valid values are 'ODD', 'NONZERO', 'POSITIVE', 'NEGATIVE' and
       * 'ABS_GEQ_TWO'.
       */
      void setWindingRule(const std::string &windingRule);

      //! Set the height of the extrusion body.
      void setHeight(float height);

      //! Delete all polygon contours.
      void clearContours();

      //! Add a new closed polygon to the base area.
      /*!
       * contour is vector of PolygonPoint 's defining the closed polygon.
       * NOTE: Memory management: see Extrution constructor.
       *
       */
      void addContour(std::vector<PolygonPoint*> *contour);

      //! Write a extrusion body file.
      virtual void writeBodyFile() const;

    protected:
      std::string windingRule;
      float height;
      std::vector<std::vector<PolygonPoint*>*> contours;

      void writeToBodyFile(std::ofstream *) const;

    private:
      bool destroyMem;
  };

}

#endif
