/*
   AMVisC++Interface - C++ Interface for Body- and Pos-File Creation
   Used by AMVis
   Copyright (C) 2006 Institute of Applied Mechanics,
   Technical University of Munich

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   */

#ifndef OBJOBJECT_H
#define OBJOBJECT_H

#include <crigidbody.h>
#include <cmath>

namespace AMVis {

  //! Creates a cube.
  /*!
   * This class is used to create a cube, which can be moved and rotation in
   * space.
   */
  class ObjObject : public CRigidBody {
    public:
      //! Create a Wavefront object body.
      /*!
       * baseFilename is the basename of the body and pos file. instances is the
       * number of instances this body should create and if binPosFile is true the
       * pos file is created in binary mode or in ascii mode if false.
       */
      ObjObject(const std::string &, int, bool);

      //! Create a Wavefront object body.
      /*!
       * Use this constructor if you want to append this body to a
       * CompoundPrimitiveBody. This constructor dose not create a body or pos
       * file.
       */
      ObjObject();

      //! Set the filename of the Wavefront object file.
      void setObjFilename(const std::string &);

      //! Use textures from obj material file.
      void setUseTextures(bool);

      //! Use materials from obj material file.
      void setUseMaterials(bool);

      //! Calculation methode for vertex normals.
      /*!
       * Use normals from obj-file if 0, calculate per face normals (flat shading)
       * if 1 and calculate per vertex normals (smooth shading) if 2. If 3 
       * calculate real normals (smooth if angle is small).
       */
      void setCalculationOfNormals(int);

      //! eps-value when vertices are to be treated as equal.
      /*! This is only used for calculation of the outline. */
      void setVertexEPS(float);

      //! eps-value when normals are to be treated as equal.
      /*! This is only used for calculation of the outline. */
      void setNormalEPS(float);

      //! eps-value when faces are to be treated as smooth.
      /*! This is only used for calculation of the real (3) normals. */
      void setAngleEPS(float);

      //! Generation methode for outlines.
      /*!
       * Generate no outline if 0, calculate outline on the fly if 1 and read
       * outline from a obj-file named '<ObjFilename>.outline.obj' if 2.
       */
      void setOutline(int);

      //! Write a Wavefront object body file.
      virtual void writeBodyFile() const;

    protected:
      std::string objFilename;
      bool useTex, useMat;
      int normalFlag, outlineFlag;
      float epsVertex, epsNormal, epsAngle;

      void writeToBodyFile(std::ofstream *) const;
  };

}

#endif
