/*
   AMVisC++Interface - C++ Interface for Body- and Pos-File Creation
   Used by AMVis
   Copyright (C) 2006 Institute of Applied Mechanics,
   Technical University of Munich

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   */

#ifndef ROTATION_H
#define ROTATION_H

#include <crigidbody.h>
#include <vector>
#include <polygonpoint.h>

namespace AMVis {

  //! Creates a rotational body.
  /*!
   * Creates a rotational body with the rotation axis y and a
   * user defined cross-sectional area given by a closed polygon.
   */
  class Rotation : public CRigidBody {
    public:
      //! Create a rotational body.
      /*!
       * baseFilename is the basename of the body and pos file. instances is the
       * number of instances this body should create and if binPosFile is true the
       * pos file is created in binary mode or in ascii mode if false.
       * If destroymem is true (default) then the destructor of this object will
       * delete all the memory allocated by a contour added with setContour.
       */
      Rotation(const std::string &, int, bool, bool=true);

      //! Create a rotational body.
      /*!
       * Use this constructor if you want to append this body to a
       * CompoundPrimitiveBody. This constructor dose not create a body of pos file
       * If destroymem is true (default) then the destructor of this object will
       * delete all the memory allocated by a contour added with setContour.
       */
      Rotation(bool=true);

      //! Delete a rotational body.
      ~Rotation();

      //! Set the cross-sectional area.
      /*!
       * contour is vector of PolygonPoint 's defining the cross-sectional area.
       * NOTE: Memory management: see Rotation constructor.
       */
      void setContour(std::vector<PolygonPoint*> *contour);

      //! Write a rotational body file.
      virtual void writeBodyFile() const;

    protected:
      std::vector<PolygonPoint*> *contour;

      void writeToBodyFile(std::ofstream *) const;


    private:
      bool destroyMem;

      void clearContour();
  };

}

#endif
