/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <arrow.h>
#include <GL/glu.h>

// Constructor
Arrow::Arrow(char* body_name, char *pos_file) : CBody(body_name, pos_file), c(0), l(0) {
  binaryrecordsize=sizeof(float)*8;
  int result;
  char* charstringline = new char [charlength];

  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&d);

  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&D);

  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&h);

  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%d" ,&m);

  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  result=sscanf(charstringline, "%f" ,&scale_factor);
  if (result<=0) scale_factor=1;				//Wegen Kompatibilität zu alten body files

  quadObj=gluNewQuadric();
  gluQuadricDrawStyle(quadObj, GLU_FILL);
}

// Destructor
Arrow::~Arrow() {
  gluDeleteQuadric(quadObj);
}

void Arrow::drawBody() {
  float h_=h, D_=D, d_=d;
  if(l<2*h) {
    h_=l/2;
    D_=D/h*h_;
    d_=d/D*D_;
  }
  if(m==0) {
    glTranslatef(0, 0, -l);
    gluDisk(quadObj, 0, d_/2, NCIRCLE(d_/2), 1);
    gluCylinder(quadObj, d_/2, d_/2, l, NCIRCLE(d_/2), 1);
    glTranslatef(0, 0, l);
    gluDisk(quadObj, 0, d_/2, NCIRCLE(d_/2), 1);
  } else if(m==1) {
    glTranslatef(0, 0, -l);
    gluCylinder(quadObj, d_/2, d_/2, l-h_, NCIRCLE(d_/2), 1);
    gluDisk(quadObj, 0, d_/2, NCIRCLE(d_/2), 1);
    glTranslatef(0, 0, l-h_);
    gluCylinder(quadObj, D_/2, 0, h_, NCIRCLE(D_/2), 1);
    gluDisk(quadObj, 0, D_/2, NCIRCLE(D_/2), 1);
    glTranslatef(0, 0, h_);
  } else if(m==2) {
    gluDisk(quadObj, 0, d_/2, NCIRCLE(d_/2), 1);
    glTranslatef(0, 0, -(l-h_));
    gluCylinder(quadObj, d_/2, d_/2, l-h_, NCIRCLE(d_/2), 1);
    gluDisk(quadObj, 0, D_/2, NCIRCLE(D_/2), 1);
    glTranslatef(0, 0, -h_);
    gluCylinder(quadObj, 0, D_/2, h_, NCIRCLE(D_/2), 1);
    glTranslatef(0, 0, l);
  } else if(m==3) {
    glTranslatef(0, 0, -l);
    gluCylinder(quadObj, 0, D_/2, h_, NCIRCLE(D_/2), 1);
    glTranslatef(0, 0, h_);
    gluDisk(quadObj, 0, D_/2, NCIRCLE(D_/2), 1);
    gluCylinder(quadObj, d_/2, d_/2, l-h_-h_, NCIRCLE(d_/2), 1);
    glTranslatef(0, 0, l-h_-h_);
    gluDisk(quadObj, 0, D_/2, NCIRCLE(D_/2), 1);
    gluCylinder(quadObj, D_/2, 0, h_, NCIRCLE(D_/2), 1);
    glTranslatef(0, 0, h_);
  }
}

// Draw body
void Arrow::draw() {
  if(l>1e-10) {
    glDisable(GL_COLOR_MATERIAL);
    glEnable(GL_LIGHTING);
    GLfloat color[4];
    translateColorvalueToRGBA(c, &color[0], &color[1], &color[2], &color[3]);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
    GLfloat mat_specular[] = {1.0, 1.0, 1.0, 1.0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
    float color_shininess=20;
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
    float em[4]={0,0,0,0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);

    glPushMatrix();
    glTranslatef(x, y, z);
    glRotatef(atan2(dy,dx)*180/M_PI, 0, 0, 1);
    glRotatef(atan2(sqrt(dx*dx+dy*dy),dz)*180/M_PI, 0, 1, 0);

    if(mode==0) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    if(mode==1) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    if(mode==2) glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);

    drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!
    
    if(contour_flag==true) {
      // draw contour
      glEnable(GL_POLYGON_OFFSET_LINE);
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glEnable(GL_COLOR_MATERIAL);
      glLineWidth(1.0);
      glPolygonOffset(OFFSETFAC1,OFFSETUNIT1);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glColor3f(0.0, 0.0, 0.0);

      float h_=h, D_=D, d_=d;
      if(l<2*h) {
        h_=l/2;
        D_=D/h*h_;
        d_=d/D*D_;
      }
      if(m==0) {
        glTranslatef(0, 0, -l);
        Circle_Line(d_/2);
        glTranslatef(0, 0, l);
        Circle_Line(d_/2);
      } else if(m==1) {
        glTranslatef(0, 0, -l);
        Circle_Line(d_/2);
        glTranslatef(0, 0, l-h_);
        Circle_Line(D_/2);
        Circle_Line(d_/2);
        glTranslatef(0, 0, h_);
      } else if(m==2) {
        Circle_Line(d_/2);
        glTranslatef(0, 0, -(l-h_));
        Circle_Line(D_/2);
        Circle_Line(d_/2);
        glTranslatef(0, 0, -h_+l);
      } else if(m==3) {
        glTranslatef(0, 0, -l+h_);
        Circle_Line(D_/2);
        Circle_Line(d_/2);
        glTranslatef(0, 0, l-h_-h_);
        Circle_Line(D_/2);
        Circle_Line(d_/2);
        glTranslatef(0, 0, h_);
      }
      
      glPolygonOffset(0.0,0.0);
    }
    
    if(selected_flag==true) {
        float min_x, max_x, min_y, max_y, min_z, max_z;
        return_max_dimensions(&min_x, &max_x, &min_y, &max_y, &min_z, &max_z);
        glEnable(GL_COLOR_MATERIAL);
        glDisable(GL_LIGHTING);
        glEnable(GL_LINE_SMOOTH);
        glColor3f(0.0, 1.0, 0.0);
        glLineWidth(4.0);
        glBegin(GL_LINE_STRIP);
        glVertex3f(min_x, min_y, min_z);
        glVertex3f(min_x, max_y, min_z);
        glVertex3f(min_x, max_y, max_z);
        glVertex3f(min_x, min_y, max_z);
        glVertex3f(min_x, min_y, min_z);
        glVertex3f(max_x, min_y, min_z);
        glVertex3f(max_x, max_y, min_z);
        glVertex3f(max_x, max_y, max_z);
        glVertex3f(max_x, min_y, max_z);
        glVertex3f(max_x, min_y, min_z);
        glEnd();
        glBegin(GL_LINES);
        glVertex3f(min_x, min_y, min_z);
        glVertex3f(max_x, min_y, min_z);
        glVertex3f(min_x, max_y, min_z);
        glVertex3f(max_x, max_y, min_z);
        glVertex3f(min_x, max_y, max_z);
        glVertex3f(max_x, max_y, max_z);
        glVertex3f(min_x, min_y, max_z);
        glVertex3f(max_x, min_y, max_z);
        glEnd();
        glEnable(GL_LIGHTING);
        glPolygonOffset(0.0,0.0);
        glEnable(GL_POLYGON_OFFSET_LINE);
        glDisable(GL_COLOR_MATERIAL);
    }
    
    if(local_cos_flag==true) {
      // draw something
    }

    glPopMatrix();
  }

  if(path_flag) {
    long drawpathto=curdataset;
    if(path.size()<=(unsigned long)drawpathto) {
      for(int i=path.size(); i<=drawpathto; i++) {
        update(i);
        Point3D *p=new Point3D;
        p->x=x;
        p->y=y;
        p->z=z;
        path.push_back(p);
      }
    }
    drawPath(0, 1, 0);
  }
}

// Initialize (Call CBody::init()!)
void Arrow::init() {
  CBody::init();
}

// Read data from pos-file
void Arrow::update(long dataset_number) {
  fseek(posfile_pointer, dataset_number*recordsize, SEEK_SET);
  if(pos_file_binary==true) {
    float dummy[8];
    fread(dummy,sizeof(float),8,posfile_pointer);
    current_time=dummy[0];
    x=dummy[1];
    y=dummy[2];
    z=dummy[3];
    dx=dummy[4];
    dy=dummy[5];
    dz=dummy[6];
    c=dummy[7];
  } else
    fscanf(posfile_pointer, "%f %f %f %f %f %f %f %f", &current_time, &x, &y, &z, &dx, &dy, &dz, &c);
  l=sqrt(dx*dx+dy*dy+dz*dz)*scale_factor;
  curdataset=dataset_number;
}

// For selection by mouse click
void Arrow::draw_pure() {
  if(l>1e-10) {
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    glDisable(GL_LIGHTING);
    glEnable(GL_COLOR_MATERIAL);
    glColor3fv(pick_color_value);
    glPushMatrix();

    glTranslatef(x, y, z);
    glRotatef(atan2(dy,dx)*180/M_PI, 0, 0, 1);
    glRotatef(atan2(sqrt(dx*dx+dy*dy),dz)*180/M_PI, 0, 1, 0);
    drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!

    glPopMatrix();
    glEnable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
  }
}

// Text in the left
void Arrow::return_info_string(QString &info) {
  info=tr("Class: %1\nBody: %2\nPos-File: %3\nDir.: %4,%5,%6\nLength: %7").arg(class_name).arg(body_file_name).arg(posfile_name).arg(dx).arg(dy).arg(dz).arg(l/scale_factor);
}

// For move camera with body (rigid-body movement)
void Arrow::ret_rot(float *rotation) {
  rotation[0]=0;
  rotation[1]=0;
  rotation[2]=0;
}

// For move camera with body (rigid-body movement)
void Arrow::ret_trans(float *translation) {
  translation[0]=x;
  translation[1]=y;
  translation[2]=z;
}

void Arrow::return_max_dimensions(float *min_x, float *max_x, float *min_y, float *max_y, float *min_z, float *max_z) {
  ////////////////////////
  *min_x=-D/2;
  *max_x=D/2;
  *min_y=-D/2;
  *max_y=D/2;
  *min_z=-l;
  *max_z=0;
}
