/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <GL/gl.h>
#include "cbody.h" // class's header file

using namespace std;

float CBody::EPS=0.000001;

// class constructor
CBody::CBody() : enabled(true), curdataset(0), olddatasetBody(-1), secondolddatasetBody(-2), olddatasetPath(-1), secondolddatasetPath(-2), curDisplList(0)
{
   // QMessageBox::information(NULL, "constructor CBody","CBody::CBody()");
   posfile_name = new char [charlength];
   class_name = new char [charlength];
   body_file_name = new char [charlength];
   local_cos_flag = false;
   contour_flag = false;
   selected_flag = false;
   enabled = true;
   pos_file_binary = false;
   pick_color_value[R]=0;
   pick_color_value[G]=0;
   pick_color_value[B]=0;
   mode =0;
   posfile_pointer=0;

}


CBody::CBody(char* body_name, char * pos_file) : enabled(true), curdataset(0), olddatasetBody(-1), secondolddatasetBody(-2), olddatasetPath(-1), secondolddatasetPath(-2), curDisplList(0)
{
   printf("AMVis: Creating body '%s'; Pos-file: '%s'\n",body_name,pos_file);
   posfile_name = new char [charlength];
   class_name = new char [charlength];
   body_file_name = new char [charlength];
   local_cos_flag = false;
   contour_flag = false;
   selected_flag = false;
   enabled = true;
   mode =0;
   strcpy(body_file_name,body_name);
   strcpy(posfile_name , pos_file);
   int result;
   char* charstringline = new char [charlength];
   datafile = fopen (body_file_name,"rb");
   //rewind the file
   fseek(datafile,0,SEEK_SET);
   //jump to Class Name
   parsenextdataline(datafile);
   result = GetCharacterLine(class_name, charlength, datafile);
   //jump to number of instances
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   //File type of position file(0=text, 1=binary)
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   int dummy;
   sscanf(charstringline, "%i\n", &dummy);
   pos_file_binary=dummy;
   posfile_pointer=0;
}

CBody::CBody(FILE *bodyfile) : enabled(true), curdataset(0), olddatasetBody(-1), secondolddatasetBody(-2), olddatasetPath(-1), secondolddatasetPath(-2), curDisplList(0)
{
   printf("AMVis: Creating sub-body\n");
   datafile=bodyfile;
   posfile_name = new char [charlength];
   class_name = new char [charlength];
   body_file_name = new char [charlength];
   local_cos_flag = false;
   contour_flag = false;
   selected_flag = false;
   enabled = true;
   mode =0;
   posfile_pointer=0;
}

// class destructor
CBody::~CBody()
{
  std::vector<Point3D*>::iterator i;
  for(i=path.begin(); i!=path.end(); i++)
    delete *i;
  path.clear();
}

void CBody::translateColorvalueToRGBA(float colorvalue, float* r, float *g, float *b, float *a)
{
   QColor NewColorValue;
   int r_color, g_color, b_color, a_color;
   int Hue_Value;
   Hue_Value=(int)(HUE_MAX_VALUE-(HUE_MAX_VALUE * colorvalue));
   NewColorValue.setHsv(Hue_Value, 255, 255, 255);
   NewColorValue.getRgb(&r_color, &g_color, &b_color, &a_color);
   *r=r_color/255.0;
   *g=g_color/255.0;
   *b=b_color/255.0;
   *a=a_color/255.0;
}

void CBody::translateRGBAToColorvalue(float r, float g, float b, float a, float *colorvalue)
{
   QColor NewColorValue;
   int h,s,v,alpha;
   int r_color, g_color, b_color, a_color;
   r_color = (int) r*255;
   g_color = (int) g*255;
   b_color = (int) b*255;
   a_color = (int) a*255;
   NewColorValue.setRgb(r_color, g_color, b_color, a_color);
   NewColorValue.getHsv(&h, &s, &v, &alpha);
   *colorvalue=(float)h/HUE_MAX_VALUE;
}

int CBody::GetCharacterLine(char *pCharacterLineBuffer, int iBufferSize, FILE *pFile)
{
   if (pCharacterLineBuffer == NULL || pFile == NULL) return Error_wrong_params;
   if (iBufferSize < 2) return Error_buffertoosmall;
   int ch, count = 0;
   long pos_old;
   while (true)
   {
      if (count == (iBufferSize-1))
      {
	 pCharacterLineBuffer[count] = '\0'; // Terminate
	 return Error_linetoolong;
      }
      ch = fgetc(pFile);
      if (ch != EOF)
      {
	 if ((ch == CR) || (ch == LF))
	 {
	    pCharacterLineBuffer[count] = '\0'; // Terminate
	    if (count==0)
	       continue; // Get next line
	    else
	    {
	       pos_old=ftell (pFile);
	       ch = fgetc(pFile);
	       if ((ch == CR) || (ch == LF)) {}else fseek (pFile,pos_old,SEEK_SET);
	       break;
	    }; // Get out - return success
	 }
	 else {
	    pCharacterLineBuffer[count] = (char)ch;
	    count++;
	 }
      }
      else {
	 pCharacterLineBuffer[count] = '\0'; // Terminate
	 if (!ferror(pFile)) return EOF;
	 return Error_readerror;
      }
   };
   return count;
}

void CBody::parsenextdataline(FILE * pfilepointer)
{
   char* newline=new char[128];
   int ret;
   long oldpos;
   while (1 )

   {
      oldpos=ftell(pfilepointer);
      newline[0] = '#';
      ret = GetCharacterLine(newline, 128, pfilepointer);
      if (ret == EOF) break; // Get out of while loop
      if( (newline[0]!='#') ) break;
   };
   fseek (pfilepointer,oldpos,SEEK_SET);
}

void CBody::CrossProduct(float* u, float* v, float* n)
{
   n[X] = u[Y] * v[Z] - u[Z] * v[Y];
   n[Y] = u[Z] * v[X] - u[X] * v[Z];
   n[Z] = u[X] * v[Y] - u[Y] * v[X];
}

void CBody::Normalize(float* n)
{
   GLfloat normal;
   normal = (float)sqrt(n[X] * n[X] + n[Y] * n[Y] + n[Z] * n[Z]);
   n[0] /= normal;
   n[1] /= normal;
   n[2] /= normal;
}

void CBody::ReduceToUnit(float vector[3])
{
   float length;

   length = (float)sqrt((vector[0]*vector[0]) +
			(vector[1]*vector[1]) +
			(vector[2]*vector[2]));
   if(length == 0.0f)
      length = 1.0f;
   vector[0] /= length;
   vector[1] /= length;
   vector[2] /= length;
}

void CBody::calcNormal(float v[3][3], float out[3])
{
   float v1[3],v2[3];
   static const int x = 0;
   static const int y = 1;
   static const int z = 2;
   v1[x] = v[0][x] - v[1][x];
   v1[y] = v[0][y] - v[1][y];
   v1[z] = v[0][z] - v[1][z];
   v2[x] = v[1][x] - v[2][x];
   v2[y] = v[1][y] - v[2][y];
   v2[z] = v[1][z] - v[2][z];
   out[x] = v1[y]*v2[z] - v1[z]*v2[y];
   out[y] = v1[z]*v2[x] - v1[x]*v2[z];
   out[z] = v1[x]*v2[y] - v1[y]*v2[x];
   ReduceToUnit(out);
}

void CBody::Cylinder(	float diameter, float height, float angle_deg_1, float angle_deg_2,  bool frontface)
{
   float x,y,angle;
   float normal[3],corners[4][3];
   float step = (PI/180*360.0/NCIRCLE(diameter));
   glFrontFace( GL_CW);
   glBegin(GL_QUADS);
   for(angle = angle_deg_1/180*PI; angle <  angle_deg_2/180*PI; angle += step)
   {
      x = diameter*(float)sin(angle);
      y = diameter*(float)cos(angle);
      corners[0][0] = x;
      corners[0][1] = y;
      corners[0][2] = -height;
      corners[1][0] = x;
      corners[1][1] = y;
      corners[1][2] = 0.0f;
      x = diameter*(float)sin(angle+step);
      y = diameter*(float)cos(angle+step);

      if(angle+step <  angle_deg_2/180*PI) // Not Finished
      {
	 corners[2][0] = x;
	 corners[2][1] = y;
	 corners[2][2] = 0.0f;

	 corners[3][0] = x;
	 corners[3][1] = y;
	 corners[3][2] = -height;
      }
      else
      {
	 x = diameter*(float)sin(angle_deg_2/180*PI);
	 y = diameter*(float)cos(angle_deg_2/180*PI);
	 corners[2][0] = x;
	 corners[2][1] = y;
	 corners[2][2] = 0.0f;
	 corners[3][0] = x;
	 corners[3][1] = y;
	 corners[3][2] = -height;
      }
  if ( frontface ==true)   
  {
       normal[X] = corners[1][0];
       normal[Y] = corners[1][1];
       normal[Z] = corners[1][2];
  }
   else 
  {
       normal[X] = -corners[1][0];
       normal[Y] = -corners[1][1];
       normal[Z] = -corners[1][2];
  }
      glNormal3fv(normal);
      glVertex3fv(corners[0]);
      glVertex3fv(corners[1]);

  if ( frontface ==true)  
  {
       normal[X] = corners[2][0];
       normal[Y] = corners[2][1];
       normal[Z] = corners[2][2];
  }
  else
  {
       normal[X] = -corners[2][0];
       normal[Y] = -corners[2][1];
       normal[Z] = -corners[2][2];
  }
      glNormal3fv(normal);
      glVertex3fv(corners[2]);
      glVertex3fv(corners[3]);
   }
   glEnd();
   glFrontFace( GL_CCW);
}

void CBody::Circle(	float diameter,  float angle_deg_1, float angle_deg_2, bool frontface)
{
   float x,y,angle;
   float corners[4][3];
   float step = (PI/180*360.0/NCIRCLE(diameter));
 
 if (frontface ==true)  glFrontFace( GL_CW);

   corners[0][0] = 0.0;
   corners[0][1] = 0.0;
   corners[0][2] = 0.0;
   glBegin(GL_TRIANGLES);
   for(angle = angle_deg_1/180*PI; angle <  angle_deg_2/180*PI; angle += step)
   {
      x = diameter*(float)sin(angle);
      y = diameter*(float)cos(angle);
      corners[1][0] = x;
      corners[1][1] = y;
      corners[1][2] = 0.0;

      if(angle+step <  angle_deg_2/180*PI) // Not Finished
      {
	 x = diameter*(float)sin(angle+step);
	 y = diameter*(float)cos(angle+step);
	 corners[2][0] = x;
	 corners[2][1] = y;
	 corners[2][2] = 0.0f;

      }
      else
      {
	 x = diameter*(float)sin(angle_deg_2/180*PI);
	 y = diameter*(float)cos(angle_deg_2/180*PI);
	 corners[2][0] = x;
	 corners[2][1] = y;
	 corners[2][2] = 0.0f;
      }

      if (frontface == true) glNormal3f( 0.0, 0.0, 1.0f ); 
      else glNormal3f( 0.0, 0.0, -1.0f );

      glVertex3fv(corners[0]);
      glVertex3fv(corners[1]);
      glVertex3fv(corners[2]);

   }
   glEnd();
   glFrontFace( GL_CCW);
}

void CBody::Circle_Line(	float diameter     )
{
   glBegin(GL_LINE_LOOP);
   for(float angle=0;angle<2.0*M_PI;angle+=2.0*M_PI/NCIRCLE(diameter))
     glVertex3f(diameter*cos(angle),diameter*sin(angle),0);
   glEnd();
}

void CBody::ret_time(float *time) {
  *time=current_time;
}

long int CBody::get_dataset_last() {
  long curpos, lastpos;
  curpos=ftell(posfile_pointer);
  fseek (posfile_pointer,0,SEEK_END);
  lastpos =ftell(posfile_pointer);
  fseek (posfile_pointer, curpos, SEEK_SET);
  return lastpos/recordsize;
}

long CBody::getrecordsizetext(FILE * textfilepointer)
{
   long oldposition, pos_before_cr, newposition;
   int ch;
   oldposition=ftell (textfilepointer);
   fseek (textfilepointer,0,SEEK_SET);
   while (true)
   {
      ch = fgetc(textfilepointer);
      if (ch != EOF)
      {
         if ((ch == CR) || (ch == LF))
         {
            pos_before_cr=ftell (textfilepointer);
            ch = fgetc(textfilepointer);
            //                             printf("Found CR / LF before %ld\n",ftell (textfilepointer));
            if ((ch == CR) || (ch == LF)) {}else
            {
               fseek (textfilepointer,pos_before_cr,SEEK_SET);
            };
            newposition=ftell (textfilepointer);
            break;
         }; // Get out - return success
      }
      else
      {
         fseek (textfilepointer,oldposition,SEEK_SET);
         return 0;
         break;
      };
   };
   fseek (textfilepointer,oldposition,SEEK_SET);
   return (newposition);
};

long int CBody::get_dataset() {
  return ftell(posfile_pointer)/recordsize;
}

void CBody::init() {
  current_time=0;
  if(posfile_pointer==0)
    posfile_pointer=fopen(posfile_name,"rb");
  if(pos_file_binary==true)
    recordsize=binaryrecordsize;
  else
    recordsize=getrecordsizetext(posfile_pointer);
}

void CBody::cachePosFile() {
  bool close=false;
  if(!posfile_pointer) {
    posfile_pointer=fopen(posfile_name,"rb");
    close=true;
  }
  int size;
  const int BUFFSIZE=1048576;
  char buff[BUFFSIZE];
  do {
    size=fread(buff, sizeof(char), BUFFSIZE, posfile_pointer);
  }
  while(size==BUFFSIZE);
  if(close) {
    fclose(posfile_pointer);
    posfile_pointer=0;
  }
}

void CBody::drawPath(float r, float g, float b) {
    glEnable(GL_POLYGON_OFFSET_LINE);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    glEnable(GL_COLOR_MATERIAL);
    glLineWidth(1.5);
    glPolygonOffset(OFFSETFAC1,OFFSETUNIT1);
    glDisable(GL_LIGHTING);
    glDisable(GL_LINE_SMOOTH);
    glColor3f(r, g, b);
  
    if(curdataset!=olddatasetPath) {
      glBegin(GL_LINE_STRIP);
        for(int i=0; i<=curdataset; i++) {
          glVertex3f(path[i]->x,path[i]->y,path[i]->z);
        }
      glEnd();
    }
    else if(olddatasetPath!=secondolddatasetPath) {
      if(curDisplListPath) glDeleteLists(curDisplListPath, 1);
      curDisplListPath=glGenLists(1);
      glNewList(curDisplListPath, GL_COMPILE_AND_EXECUTE);

      glBegin(GL_LINE_STRIP);
        for(int i=0; i<=curdataset; i++) {
          glVertex3f(path[i]->x,path[i]->y,path[i]->z);
        }
      glEnd();

      glEndList();
    }
    else
      glCallList(curDisplListPath);
    secondolddatasetPath=olddatasetPath;
    olddatasetPath=curdataset;
  
    glPolygonOffset(0.0,0.0);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    glDisable(GL_POLYGON_OFFSET_LINE);
}

vector<CBody::Contour*> CBody::recalcContour(vector<CBody::Contour*> &contour) {
  if(contour.size()<=0) return contour;

  unsigned int n, m, i;
  float nx, ny, mx, my, ix, iy;
  float g=0;
  vector<Contour*> newcont;
  unsigned int lastPushed;

  n=0; m=0;
  if(newcont.size()<=n) { newcont.push_back(contour[n]); lastPushed=n; }
  while(n<contour.size() && m<contour.size()) {
    if(n+1<contour.size() && contour[n+1]->b==1) {
      n++;
      if(newcont.size()<=n) { newcont.push_back(contour[n]); lastPushed=n; }
      continue;
    }
    nx=contour[n]->x; ny=contour[n]->y;
    for(m=n+2; m<contour.size(); m++) {
      mx=contour[m]->x; my=contour[m]->y;
      for(i=n+1; i<m; i++) {
        ix=contour[i]->x; iy=contour[i]->y;
        g=sqrt((-ny-((iy-ny)*(my-ny)+(ix-nx)*(mx-nx))*(my-ny)/((my-ny)*(my-ny)+(mx-nx)*(mx-nx))+iy)*(-ny-((iy-ny)*(my-ny)+(ix-nx)*(mx-nx))*(my-ny)/((my-ny)*(my-ny)+(mx-nx)*(mx-nx))+iy)+(-(mx-nx)*((iy-ny)*(my-ny)+(ix-nx)*(mx-nx))/((my-ny)*(my-ny)+(mx-nx)*(mx-nx))-nx+ix)*(-(mx-nx)*((iy-ny)*(my-ny)+(ix-nx)*(mx-nx))/((my-ny)*(my-ny)+(mx-nx)*(mx-nx))-nx+ix));
        if(g>EPS) break;
      }
      if(g>EPS) {
        if(newcont.size()<=m-1) { newcont.push_back(contour[m-1]); lastPushed=m-1; }
        n=m-1;
        break;
      }
      if(contour[m]->b==1) {
        if(newcont.size()<=m) { newcont.push_back(contour[m]); lastPushed=m; }
        n=m;
        break;
      }
    }
  }
  if(newcont.size()<=contour.size()-1 && lastPushed!=contour.size()-1)
    newcont.push_back(contour[contour.size()-1]);
  return newcont;
}

void CBody::drawBodyFast() {
  if(curdataset!=olddatasetBody)
    drawBody();
  else if(olddatasetBody!=secondolddatasetBody) {
    if(curDisplList) glDeleteLists(curDisplList, 1);
    curDisplList=glGenLists(1);
    glNewList(curDisplList, GL_COMPILE_AND_EXECUTE);
    drawBody();
    glEndList();
  }
  else
    glCallList(curDisplList);
  secondolddatasetBody=olddatasetBody;
  olddatasetBody=curdataset;
}
