/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef CBODY_H
#define CBODY_H


#include <string>
#include "GL/glu.h"
#include "GL/gl.h"
#include <math.h>
#include <stdio.h>
#include <cstdlib>
#include <iostream>
#include <QColor>
#include <QMessageBox>
#include <vector>
#include <QMenu>

#define charlength 10000
// Anzahl der Polygonecken einer Kreisapprox mit radius r und max. Fehler EPS
#define NCIRCLE(r) ((int(M_PI/acos(1-EPS/(r))/4)+1)*4)
#define OFFSETFAC0 1
#define OFFSETFAC1 1
#define OFFSETFAC2 1
#define OFFSETFACM3 1
#define OFFSETUNIT0 0
#define OFFSETUNIT1 1
#define OFFSETUNIT2 2
#define OFFSETUNITM3 -3

#define CR 0x0D // Cariage Return
#define LF 0x0A // Line Feed
#define Error_wrong_params (-2)
#define Error_buffertoosmall (-3)
#define Error_readerror (-4)
#define Error_linetoolong (-5)
#define HUE_MAX_VALUE 255
#define PI 3.1415926535897932384626433832795

enum { X, Y, Z, W };
enum { R, G, B};

struct Point3D {
  float x, y, z;
};

class CBody : public QObject
{
  Q_OBJECT

protected:
  struct Contour {
    float x, y;
    int b;
  };

  std::vector<Contour*> recalcContour(std::vector<Contour*> &contour);
  

public:
   // class constructor
   CBody();
   CBody(char* body_name, char * pos_file);
   CBody(FILE *bodyfile);
   // class destructor
   virtual ~CBody();
   char* posfile_name;
   char* class_name;
   char* body_file_name;
   bool enabled;
   bool contour_flag;
   bool local_cos_flag;
   bool path_flag;
   bool selected_flag;
   int mode;
   FILE * datafile;
   bool pos_file_binary;
   float pick_color_value[3];

/**
 * Virtual function for reading data, not yet used, but probably
 * useful for further enhancements
 */
   virtual void read_data() {}

/**
 * Virtual draw function. By calling this function, the object should draw
 * itself completely; Hided objects are not getting renderd (FW)
 */
   virtual void draw() = 0;

/**
 * Virtual init function. This function is called after the object is created.
 * Useful for setting special parameters
 */
   virtual void init();

/**
 * get_dataset should return the objects curret dataset. If it is an object with
 * a text base position file, this means the current line number
 */
   virtual long get_dataset();

/**
 * get_daatset_last should return the last dataset.
 * If it is an object with a text base position file, this means the last line number
 * Or the record before EOF (end of file)
 */
   virtual long get_dataset_last();

/**
 * update should force the object, to update its position by the given dataset - number
 * If it is based on a text - pos file. Read line x  ( x =  dataset_number) and update all read values
 * for example position , colour etc.
 */
   virtual void update(long dataset_number) = 0;

/**
 * This function should return the minimal and maximum dimensions of an object
 */
   virtual void return_max_dimensions(float* min_x, float* max_x, float* min_y, float* max_y,float* min_z, float* max_z) {
     *min_x=-0.1;
     *max_x=0.1;
     *min_y=-0.1;
     *max_y=0.1;
     *min_z=-0.1;
     *max_z=0.1;
   };

/**
 * ret_time should return the current time of the object, which corresponds to the last time
 * read in the position file
 */
   virtual void ret_time(float* time);

/**
 * ret_time should return the current rotation angles (float[3]) of the object, which corresponds to the last angles
 * read in the position file
 */
   virtual void ret_rot(float* rotation) {
     rotation[0]=0;
     rotation[1]=0;
     rotation[2]=0;
   }

/**
 * ret_time should return the current tranlations (float[3]) of the object, which corresponds to the last angles
 * read in the position file
 */
   virtual void ret_trans(float* translation) {
     translation[0]=0;
     translation[1]=0;
     translation[2]=0;
   }

/**
 * draw_pure is necessary for picking the object. It should draw the object with the pick_color_value with lights disabled
 */
   virtual void draw_pure() = 0;

/**
 * return_info_string could return any QString- (including \n line breaks) for displaying it in the info box
 */
   virtual void  return_info_string(QString &info) = 0;

   int  GetCharacterLine(char *pCharacterLineBuffer, int iBufferSize, FILE *pFile);
   void parsenextdataline(FILE * pfilepointer);
   void Normalize(float* n);
   void CrossProduct(float* u, float* v, float* n);
   void translateColorvalueToRGBA(float colorvalue, float* r, float *g, float *b, float *a);
   void translateRGBAToColorvalue(float r, float g, float b, float a,float *colorvalue);
   void ReduceToUnit(float vector[3]);
   void calcNormal(float v[3][3], float out[3]);
   void Cylinder(	float diameter, float height, float angle_deg_1, float angle_deg_2, bool frontface);
   void Circle(	float diameter,  float angle_deg_1, float angle_deg_2, bool frontface);
   void Circle_Line(	float diameter     );
   
   void cachePosFile();

   virtual void edit() {};
   static void setEPS(float eps) { EPS=eps; }

protected:
   static float EPS;
   float current_time;
   FILE *posfile_pointer;
   int recordsize;
   int binaryrecordsize;
   long getrecordsizetext(FILE * textfilepointer);
   long curdataset, olddatasetBody, secondolddatasetBody, olddatasetPath, secondolddatasetPath;
   std::vector<Point3D*> path;
   void drawPath(float, float, float);

   GLuint curDisplList, curDisplListPath;
   void drawBodyFast();
   virtual void drawBody() {};

signals:
   void sigUpdateGL();
};

#endif // CBODY_H
