/*
   AMVis - 3D Multibody Visualisation Program.
   Copyright (C) 2006 Institute of Applied Mechanics,
   Technische Universitaet Muenchen

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <GL/gl.h>
#include "coilspring.h" // class's header file

// class constructor
CoilSpring::CoilSpring(char* body_name, char * pos_file): CBody( body_name,  pos_file)
{
  binaryrecordsize=sizeof(float)*8;   
  int result;
  char* charstringline = new char [charlength];

  parsenextdataline(datafile);
  result = GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f \n" ,&fWindungen);

  parsenextdataline(datafile);
  result = GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f \n" ,&R);

  parsenextdataline(datafile);
  result = GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f \n" ,&r);

  parsenextdataline(datafile);
  result = GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f \n" ,&scale_factor);
}

CoilSpring::~CoilSpring()
{
}

void CoilSpring::drawBody()
{
    fCircStep = 360.0 / iCircSegments;
    fSprStep = (fWindungen * 360.0) / iSprSegments;
    theta = (float)atan(fLaenge / (fWindungen*2.0*M_PI*R)); // Steigungswinkel
    fZStep = fLaenge / iSprSegments;
    phi = 0.0;
    z = 0.0;
    sw = sin(theta / 180.0 * M_PI);
    cw = cos(theta / 180.0 * M_PI);

    glBegin(GL_TRIANGLES); //Mit GL_TRIANGLES probieren
    do
    {
      for(alpha = 0; alpha <= 360.0; alpha+=fCircStep)
      {
        v=(phi/180.0*M_PI);
        u=(alpha/180.0*M_PI);

        x = R * cos(v);
        y = R * sin(v);
        su = sin(u);
        cu = cos(u);
        sv = sin(v);
        cv = cos(v);

        //Punkt 1
        vert[0][0] = x + r * su * cv + r * cu * sv * sw;
        vert[0][1] = y - r * cu * sw * cv + r * su * sv;
        if(z + r * cu * cw < 0)
          vert[0][2] = 0;
        else if(z + r * cu * cw > fLaenge)
          vert[0][2] = fLaenge;
        else
          vert[0][2] = z + r * cu * cw;

        su = sin(u + fCircStep / 180.0 * M_PI);
        cu = cos(u + fCircStep / 180.0 * M_PI);

        //Punkt 2
        vert[1][0] = x + r * su * cv + r * cu * sv * sw;
        vert[1][1] = y - r * cu * sw * cv + r * su * sv;
        if(z + r * cu * cw < 0)
          vert[1][2] = 0;
        else if(z + r * cu * cw > fLaenge)
          vert[1][2] = fLaenge;
        else
          vert[1][2] = z + r * cu * cw;

        x = R * cos(v + fSprStep / 180.0 * M_PI);
        y = R * sin(v + fSprStep / 180.0 * M_PI);

        sv = sin(v + fSprStep / 180.0 * M_PI);
        cv = cos(v + fSprStep / 180.0 * M_PI);
        //Punkt 3
        vert[2][0] = x + r * su * cv + r * cu * sv * sw;
        vert[2][1] = y - r * cu * sw * cv + r * su * sv;
        if(z + fZStep + r * cu * cw < 0)
          vert[2][2] = 0;
        else if(z + fZStep + r * cu * cw > fLaenge)
          vert[2][2] = fLaenge;
        else
          vert[2][2] = z + fZStep + r * cu * cw;			

        su = sin(u);
        cu = cos(u);
        //Punkt 4
        vert[3][0] = x + r * su * cv + r * cu * sv * sw;
        vert[3][1] = y - r * cu * sw * cv + r * su * sv;
        if(z + fZStep + r * cu * cw < 0)
          vert[3][2] = 0;
        else if(z + fZStep + r * cu * cw > fLaenge)
          vert[3][2] = fLaenge;
        else
          vert[3][2] = z + fZStep + r * cu * cw;

        calcNormal(vert, normal);
        glNormal3f(normal[0], normal[1], normal[2]);

        //Punkte fest, ab hier zeichnen:
        glVertex3f(vert[1][0], vert[1][1], vert[1][2]);
        glVertex3f(vert[0][0], vert[0][1], vert[0][2]);
        glVertex3f(vert[2][0], vert[2][1], vert[2][2]);

        glVertex3f(vert[3][0], vert[3][1], vert[3][2]);
        glVertex3f(vert[2][0], vert[2][1], vert[2][2]);
        glVertex3f(vert[0][0], vert[0][1], vert[0][2]);
      }
      phi = phi + fSprStep;
      z = z + fZStep;
    }
    while(phi < 360.0 * fWindungen);
    glEnd();
}

void CoilSpring::draw()
{
  glDisable(GL_COLOR_MATERIAL);
  glEnable(GL_LIGHTING);
  GLfloat color[4];
  translateColorvalueToRGBA(c, &color[0], &color[1], &color[2], &color[3]);
  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
  GLfloat mat_specular[] = {1.0, 1.0, 1.0, 1.0};
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
  float color_shininess=20;
  glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
  float em[4]={0,0,0,0};
  glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);	
  // Transformation des Koordinatensystems
  // Anfangspunkt wird in P1 verschoben.
  // z-Komponente wird in Richtung des Vectors P2-P1 transformiert.
  glPushMatrix();
  glTranslatef(P1[0], P1[1], P1[2]);
  glRotatef(atan2(sqrt(d[0]*d[0] + d[1]*d[1]), d[2]) * 180.0 / M_PI, -d[1], d[0], 0.0);

  if(mode==0)
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  if(mode==1)
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  if(mode==2)
    glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);

  drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!

  if(selected_flag==true) {
    float min_x, max_x, min_y, max_y, min_z, max_z;
    return_max_dimensions(&min_x, &max_x, &min_y, &max_y, &min_z, &max_z);
    glEnable(GL_COLOR_MATERIAL);
    glDisable(GL_LIGHTING);
    glEnable(GL_LINE_SMOOTH);
    glColor3f(0.0, 1.0, 0.0);
    glLineWidth(4.0);
    glBegin(GL_LINE_STRIP);
    glVertex3f(min_x, min_y, min_z);
    glVertex3f(min_x, max_y, min_z);
    glVertex3f(min_x, max_y, max_z);
    glVertex3f(min_x, min_y, max_z);
    glVertex3f(min_x, min_y, min_z);
    glVertex3f(max_x, min_y, min_z);
    glVertex3f(max_x, max_y, min_z);
    glVertex3f(max_x, max_y, max_z);
    glVertex3f(max_x, min_y, max_z);
    glVertex3f(max_x, min_y, min_z);
    glEnd();
    glBegin(GL_LINES);
    glVertex3f(min_x, min_y, min_z);
    glVertex3f(max_x, min_y, min_z);
    glVertex3f(min_x, max_y, min_z);
    glVertex3f(max_x, max_y, min_z);
    glVertex3f(min_x, max_y, max_z);
    glVertex3f(max_x, max_y, max_z);
    glVertex3f(min_x, min_y, max_z);
    glVertex3f(max_x, min_y, max_z);
    glEnd();
    glEnable(GL_LIGHTING);
    glPolygonOffset(0.0,0.0);
    glEnable(GL_POLYGON_OFFSET_LINE);
    glDisable(GL_COLOR_MATERIAL);
    glPolygonOffset(0.0,0.0);
  }

  glPopMatrix();

  if(path_flag) {
    long drawpathto=curdataset;
    if(path.size()<=(unsigned long)drawpathto) {
      for(int i=path.size(); i<=drawpathto; i++) {
	update(i);
	Point3D *p=new Point3D;
	p->x=P1[0];
	p->y=P2[1];
	p->z=P2[2];
	path.push_back(p);
      }
    }
    drawPath(0, 1, 0);
  }

}

void CoilSpring::init() {
  CBody::init();
}

// Read data from pos-file
void CoilSpring::update(long dataset_number)
{
  fseek(posfile_pointer, dataset_number*recordsize, SEEK_SET);
  if(pos_file_binary==true) {
    float dummy[8];
    fread(dummy,sizeof(float),8,posfile_pointer);
    current_time=dummy[0];
    P1[0]=dummy[1];
    P1[1]=dummy[2];
    P1[2]=dummy[3];
    P2[0]=dummy[4];
    P2[1]=dummy[5];
    P2[2]=dummy[6];
    c=dummy[7];
  } else
    fscanf(posfile_pointer, "%f %f %f %f %f %f %f %f", &current_time, &P1[0], &P1[1], &P1[2], &P2[0], &P2[1], &P2[2], &c);
  // Berechnung der Gesamtlänge des Feders:
  d[0] = P2[0] - P1[0];
  d[1] = P2[1] - P1[1];
  d[2] = P2[2] - P1[2];
  fLaenge = sqrt(d[0]*d[0] + d[1]*d[1] + d[2]*d[2])*scale_factor;
  curdataset=dataset_number;
}

void CoilSpring::draw_pure()
{
  if(fLaenge>1e-10) {
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    glDisable(GL_LIGHTING);
    glEnable(GL_COLOR_MATERIAL);
    //    glColor3ubv(pick_color_value);
    glColor3fv(pick_color_value);
    glPushMatrix();

    glTranslatef(P1[0], P1[1], P1[2]);
    glRotatef(atan2(sqrt(d[0]*d[0] + d[1]*d[1]), d[2]) * 180.0 / M_PI, -d[1], d[0], 0.0);
    drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!

    glPopMatrix();
    glEnable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
  }
}

// Text in the left
void CoilSpring::return_info_string(QString &info)
{
  info=tr("Class: %1\nBody: %2\nPos-File: %3\nDir.: %4,%5,%6\nLength: %7\nScale: %8").arg(class_name).arg(body_file_name).arg(posfile_name).arg(P2[0]).arg(P2[1]).arg(P2[2]).arg(fLaenge/scale_factor).arg(scale_factor);
}
/*
// For move camera with body (rigid-body movement)
void CoilSpring::ret_rot(float *rotation)
{
rotation[0]=0;
rotation[1]=0;
rotation[2]=0;
}
 */
// For move camera with body
void CoilSpring::ret_trans(float *translation)
{
  translation[0]=P1[0];
  translation[1]=P1[1];
  translation[2]=P1[2];
}

void CoilSpring::return_max_dimensions(float *min_x, float *max_x, float *min_y, float *max_y, float *min_z, float *max_z)
{
  *min_x=-(R + r);
  *max_x=R + r;
  *min_y=-(R + r);
  *max_y=R + r;
  *min_z=0.0;
  *max_z=fLaenge;
}
