/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <GL/gl.h>
#include <qwt_slider.h>
#include <qwt_wheel.h>
#include <qwt_knob.h>
#include <qwt_scale_div.h>
#include <QGridLayout>
#include <QLabel>
#include <QRadioButton>
#include <QDoubleSpinBox>
#include <QCheckBox>
#include <simplematvec.h>
#include "crigidbody.h" // class's header file
#ifndef CALLBACK
#define CALLBACK
#endif

void CALLBACK beginCallbackag(GLenum which)
{
   glBegin(which);
}

void  CALLBACK errorCallbackag(GLenum errorCode)
{
  printf("Error in Tesselation. Error code: %d\n",errorCode);
}

void CALLBACK endCallbackag(void)
{
   glEnd();
}

void CALLBACK vertexCallbackag(GLdouble *vertex)
{
   glVertex3dv(vertex);
}

void CALLBACK combineCallbackag(GLdouble coords[3],
				 GLdouble *vertex_data[4],
				 GLfloat weight[4], GLdouble **dataOut )
{
   GLdouble *vertex;
   vertex = (GLdouble *) malloc(7 * sizeof(GLdouble));
   vertex[0] = coords[0];
   vertex[1] = coords[1];
   vertex[2] = coords[2];
   vertex[3] = 0.0;
   vertex[4] = 0.0;
   vertex[5] = 0.0;
   vertex[6] = 0.0;
   *dataOut = vertex;
   void *dummy; dummy=vertex_data; dummy=weight; // to prevent warning of unused parameter
}

/*
// Dose not work with Mesa
void CALLBACK edgeFlagData(
			    GLboolean flag,
			    void* polygon_data)
{
   if (flag == true) glEdgeFlag(GL_TRUE); else glEdgeFlag(GL_FALSE);
   void *dummy; dummy=polygon_data; // to prevent warning of unused parameter
};
*/

// class constructor

CRigidBody::CRigidBody(char* body_name, char * pos_file) : CBody( body_name,  pos_file), isSubBody(false), editDialog(NULL)
{
   int result;
   char* charstringline = new char [charlength];
   position_x  = 0.0;
   position_y  = 0.0;
   position_z  = 0.0;
   angle_alpha = 0.0;
   angle_beta  = 0.0;
   angle_gamma = 0.0;
   angle_alpha_initial = 0.0;
   angle_beta_initial  = 0.0;
   angle_gamma_initial = 0.0;
   //setting the initial color to red
   color[0]     =  1.0;
   color[1]     =  0.0;
   color[2]     =  0.0;
   color[3]     =  1.0;
   colorvalue = 0.0;
   color_shininess = 20.0;
   draw_color=true;
   scale_factor=1.0;
   current_time = 0.0;
   x_min = 100000000000.0;
   x_max = -100000000000.0;
   y_min = 100000000000.0;
   y_max = -100000000000.0;
   z_min = 100000000000.0;
   z_max = -100000000000.0;
   posfile_ready = false;
   recordsize = 0;
   posfile_pointer = NULL;
   //jump to Initial Translation
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   sscanf(charstringline, "%f %f %f\n" ,&position_x_initial, &position_y_initial, &position_z_initial);
   //jump to Initial Rotation
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   sscanf(charstringline, "%f %f %f\n" ,&angle_alpha_initial, &angle_beta_initial, &angle_gamma_initial);
   //jump to Initial scale factor
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   sscanf(charstringline, "%f \n" ,&scale_factor);
   Displ_List =0;

}

CRigidBody::CRigidBody(FILE *bodyfile) : CBody(bodyfile), isSubBody(true), editDialog(NULL)
{
   int result;
   char* charstringline = new char [charlength];
   position_x  = 0.0;
   position_y  = 0.0;
   position_z  = 0.0;
   angle_alpha = 0.0;
   angle_beta  = 0.0;
   angle_gamma = 0.0;
   angle_alpha_initial = 0.0;
   angle_beta_initial  = 0.0;
   angle_gamma_initial = 0.0;
   //setting the initial color to red
   color[0]     =  1.0;
   color[1]     =  0.0;
   color[2]     =  0.0;
   color[3]     =  1.0;
   colorvalue = 0.0;
   color_shininess = 20.0;
   draw_color=true;
   scale_factor=1.0;
   current_time = 0.0;
   x_min = 100000000000.0;
   x_max = -100000000000.0;
   y_min = 100000000000.0;
   y_max = -100000000000.0;
   z_min = 100000000000.0;
   z_max = -100000000000.0;
   posfile_ready = false;
   recordsize = 0;
   posfile_pointer = NULL;
   //jump to Initial Translation
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   sscanf(charstringline, "%f %f %f\n" ,&position_x_initial, &position_y_initial, &position_z_initial);
   //jump to Initial Rotation
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   sscanf(charstringline, "%f %f %f\n" ,&angle_alpha_initial, &angle_beta_initial, &angle_gamma_initial);
   //jump to Initial scale factor
   parsenextdataline(datafile);
   result = GetCharacterLine(charstringline, charlength, datafile);
   sscanf(charstringline, "%f \n" ,&scale_factor);
   Displ_List =0;

}

// class destructor
CRigidBody::~CRigidBody()
{
   if (Displ_List) glDeleteLists(Displ_List, 1);
}

void CRigidBody::draw()
{
   glLineWidth(1.0);
   glPointSize(2.0);
   glEnable(GL_LINE_SMOOTH);
   glEnable(GL_POINT_SMOOTH);
   if (mode ==0) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   if (mode ==1) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   if (mode ==2) glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);

   glDisable(GL_COLOR_MATERIAL);
   glEnable(GL_LIGHTING);
   updateRotationMatrix();
   createInitialRotationMatrix();
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glPolygonOffset(OFFSETFAC0,OFFSETUNIT0);
   glEnable(GL_POLYGON_OFFSET_FILL);
   glEnable(GL_POLYGON_OFFSET_LINE);
   glEnable(GL_POLYGON_OFFSET_POINT);
   glEnable(GL_BLEND);
   glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

   //global translation and rotation
   glTranslated(position_x, position_y, position_z);
   glMultMatrixf(object_rotation_matrix);

   if (local_cos_flag==true) // for R system
   {
      float abs=fmax(fmax(x_max,y_max),z_max);

      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glLineWidth(3.0);
      // X DIR red
      glColor3ub(90, 0, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(abs* 1.5, 0.0, 0.0);
      glEnd();
      // Y DIR green
      glColor3ub(0, 90, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, abs* 1.5, 0.0);
      glEnd();
      // Z DOR blue
      glColor3ub(0, 0, 90);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, 0.0, abs* 1.5);
      glEnd();
      glEnable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
      glLineWidth(1.0);
   };

   translateColorvalueToRGBA(colorvalue, &color[0], &color[1], &color[2], &color[3]);
   glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
   GLfloat mat_specular[] = {1.0,1.0,1.0, 1.0};
   glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
   float em[4]={0,0,0,0};
   glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);

   //initial translation and roation
   glTranslated(position_x_initial, position_y_initial, position_z_initial);
   glMultMatrixf(object_initial_matrix);
   glScaled(scale_factor, scale_factor, scale_factor);
   glCallList(Displ_List);

   glPolygonOffset(0.0,0.0);
   if (contour_flag==true)
   {
      draw_contour();
   };

   if (selected_flag==true)
   {
      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glColor3f(0.0, 1.0, 0.0);
      glLineWidth(4.0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_min, y_max, z_min);
      glVertex3f(x_min, y_max, z_max);
      glVertex3f(x_min, y_min, z_max);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_max, y_min, z_min);
      glVertex3f(x_max, y_max, z_min);
      glVertex3f(x_max, y_max, z_max);
      glVertex3f(x_max, y_min, z_max);
      glVertex3f(x_max, y_min, z_min);
      glEnd();
      glBegin(GL_LINES);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_max, y_min, z_min);
      glVertex3f(x_min, y_max, z_min);
      glVertex3f(x_max, y_max, z_min);
      glVertex3f(x_min, y_max, z_max);
      glVertex3f(x_max, y_max, z_max);
      glVertex3f(x_min, y_min, z_max);
      glVertex3f(x_max, y_min, z_max);
      glEnd();
      glEnable(GL_LIGHTING);
      glPolygonOffset(0.0,0.0);
      glEnable(GL_POLYGON_OFFSET_LINE);
      glDisable(GL_COLOR_MATERIAL);
   };


   if (local_cos_flag==true) // for L system
   {
      float abs=fmax(fmax(x_max,y_max),z_max);

      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glLineWidth(3.0);
      // X DIR red
      glColor3ub(255, 0, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(abs* 1.5, 0.0, 0.0);
      glEnd();
      // Y DIR green
      glColor3ub(0, 255, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, abs* 1.5, 0.0);
      glEnd();
      // Z DOR blue
      glColor3ub(0, 0, 255);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, 0.0, abs* 1.5);
      glEnd();
      glEnable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
   };

   glDisable(GL_POLYGON_OFFSET_FILL);
   glDisable(GL_POLYGON_OFFSET_LINE);
   glDisable(GL_POLYGON_OFFSET_POINT);
   glPolygonOffset(0.0,0.0);
   glPopMatrix();

  if(path_flag) {
    long drawpathto=curdataset;
    if(path.size()<=(unsigned long)drawpathto) {
      for(int i=path.size(); i<=drawpathto; i++) {
        update(i);
        Point3D *p=new Point3D;
        p->x=position_x;
        p->y=position_y;
        p->z=position_z;
        path.push_back(p);
      }
    }
    drawPath(0, 1, 0);
  }
}

void CRigidBody::updateminmax(float vector[3])
{
   if   (vector[0]<x_min) x_min = vector[0];
   if   (vector[0]>x_max) x_max = vector[0];
   if   (vector[1]<y_min) y_min = vector[1];
   if   (vector[1]>y_max) y_max = vector[1];
   if   (vector[2]<z_min) z_min = vector[2];
   if   (vector[2]>z_max) z_max = vector[2];
};


void CRigidBody::updateminmax(double vector[3])
{
   if   (vector[0]<x_min) x_min = vector[0];
   if   (vector[0]>x_max) x_max = vector[0];
   if   (vector[1]<y_min) y_min = vector[1];
   if   (vector[1]>y_max) y_max = vector[1];
   if   (vector[2]<z_min) z_min = vector[2];
   if   (vector[2]>z_max) z_max = vector[2];
}

void CRigidBody::updateRotationMatrix()
{
   object_rotation_matrix[0]  = cos (angle_beta)  * cos (angle_gamma) ;
   object_rotation_matrix[1]  = cos (angle_alpha) * sin (angle_gamma) + sin (angle_alpha) * sin (angle_beta) * cos(angle_gamma);  ;
   object_rotation_matrix[2]  = sin (angle_alpha) * sin (angle_gamma) - cos (angle_alpha) * sin (angle_beta)*cos(angle_gamma) ;
   object_rotation_matrix[3]  = 0.0 ;
   object_rotation_matrix[4]  =-cos (angle_beta)  * sin (angle_gamma) ;
   object_rotation_matrix[5]  = cos (angle_alpha) * cos (angle_gamma) - sin (angle_alpha) * sin (angle_beta) * sin (angle_gamma) ;
   object_rotation_matrix[6]  = sin (angle_alpha) * cos (angle_gamma) + cos(angle_alpha) * sin (angle_beta) * sin (angle_gamma) ;
   object_rotation_matrix[7]  = 0.0 ;
   object_rotation_matrix[8]  = sin (angle_beta) ;
   object_rotation_matrix[9]  =-sin (angle_alpha) * cos (angle_beta);
   object_rotation_matrix[10] = cos (angle_alpha) * cos (angle_beta);
   object_rotation_matrix[11] = 0.0 ;
   object_rotation_matrix[12] = 0.0 ;
   object_rotation_matrix[13] = 0.0 ;
   object_rotation_matrix[14] = 0.0 ;
   object_rotation_matrix[15] = 1.0 ;
}

void CRigidBody::createInitialRotationMatrix()
{
   object_initial_matrix[0]  = cos (angle_beta_initial)  * cos (angle_gamma_initial) ;
   object_initial_matrix[1]  = cos (angle_alpha_initial) * sin (angle_gamma_initial) + sin (angle_alpha_initial) * sin (angle_beta_initial) * cos(angle_gamma_initial);  ;
   object_initial_matrix[2]  = sin (angle_alpha_initial) * sin (angle_gamma_initial) - cos (angle_alpha_initial) * sin (angle_beta_initial)*cos(angle_gamma_initial) ;
   object_initial_matrix[3]  = 0.0 ;
   object_initial_matrix[4]  =-cos (angle_beta_initial)  * sin (angle_gamma_initial) ;
   object_initial_matrix[5]  = cos (angle_alpha_initial) * cos (angle_gamma_initial) - sin (angle_alpha_initial) * sin (angle_beta_initial) * sin (angle_gamma_initial) ;
   object_initial_matrix[6]  = sin (angle_alpha_initial) * cos (angle_gamma_initial) + cos(angle_alpha_initial) * sin (angle_beta_initial) * sin (angle_gamma_initial) ;
   object_initial_matrix[7]  = 0.0 ;
   object_initial_matrix[8]  = sin (angle_beta_initial) ;
   object_initial_matrix[9]  =-sin (angle_alpha_initial) * cos (angle_beta_initial);
   object_initial_matrix[10] = cos (angle_alpha_initial) * cos (angle_beta_initial);
   object_initial_matrix[11] = 0.0 ;
   object_initial_matrix[12] = 0.0 ;
   object_initial_matrix[13] = 0.0 ;
   object_initial_matrix[14] = 0.0 ;
   object_initial_matrix[15] = 1.0 ;
}

void CRigidBody::read_data()
{
}

void CRigidBody::update(long dataset_number)
{
   int erg;
   if (posfile_ready==false)getFilePointerandRecSize();
   if (posfile_ready==true)
   {

      fseek (posfile_pointer, dataset_number * recordsize ,SEEK_SET);
      if (pos_file_binary==false)
      {

	 char szLineBuffer[charlength];
	 erg=GetCharacterLine(szLineBuffer, charlength, posfile_pointer);
	 // QMessageBox::information(NULL, "void CRigidBody::update(long dataset_number)",szLineBuffer);
	 if (erg>0)
	 {
	    sscanf(szLineBuffer, "%f %f %f %f %f %f %f %f\n",
		   &current_time,
		   &position_x,
		   &position_y,
		   &position_z,
		   &angle_alpha,
		   &angle_beta,
		   &angle_gamma,
		   &colorvalue);
	 };
      }
      else //binary file
      {
	 float tempfloat[8];
	 fread (tempfloat,sizeof(float),8, posfile_pointer);
	 current_time   =tempfloat[0];
	 position_x      =tempfloat[1];
	 position_y  =tempfloat[2];
	 position_z  =tempfloat[3];
	 angle_alpha  =tempfloat[4];
	 angle_beta   =tempfloat[5];
	 angle_gamma  =tempfloat[6];
	 colorvalue =tempfloat[7];
      };//binary file
   };
   curdataset=dataset_number;
}

long CRigidBody::get_dataset()
{
   long recordsizereturn;
   if (posfile_ready==false)getFilePointerandRecSize();
   if (posfile_ready==true)
   {
      recordsizereturn = (ftell(posfile_pointer)/recordsize) ;
      return recordsizereturn;
   }else
   {
      return 0;
   };

}

long CRigidBody::get_dataset_last()
{
   long recordsizereturn;
   if (posfile_ready==false)getFilePointerandRecSize();

   if (posfile_ready==true)
   {
      recordsizereturn = (getlastposition(posfile_pointer)/recordsize) ;
      return recordsizereturn;

   }else
   {
      return 0;
   };
}

long CRigidBody::getrecordsizetext(FILE * textfilepointer)
{
   long oldposition, pos_before_cr, newposition;
   int ch;
   oldposition=ftell (textfilepointer);
   fseek (textfilepointer,0,SEEK_SET);
   while (true)
   {
      ch = fgetc(textfilepointer);
      if (ch != EOF)
      {
	 if ((ch == CR) || (ch == LF))
	 {
	    pos_before_cr=ftell (textfilepointer);
	    ch = fgetc(textfilepointer);
	    //                             printf("Found CR / LF before %ld\n",ftell (textfilepointer));
	    if ((ch == CR) || (ch == LF)) {}else
	    {
	       fseek (textfilepointer,pos_before_cr,SEEK_SET);
	    };
	    newposition=ftell (textfilepointer);
	    break;
	 }; // Get out - return success
      }
      else
      {
	 fseek (textfilepointer,oldposition,SEEK_SET);
	 return 0;
	 break;
      };
   };
   fseek (textfilepointer,oldposition,SEEK_SET);
   return (newposition);
};

long CRigidBody::getlastposition(FILE * filestream)
{
   long curpos;
   long lastpos;
   curpos=ftell(filestream);
   fseek (filestream,0,SEEK_END);
   lastpos =ftell(filestream);
   fseek (filestream, curpos, SEEK_SET);
   return lastpos;
};

bool CRigidBody::getFilePointerandRecSize()
{
   posfile_ready=true;

   FILE * tempfilepointer;
   if(posfile_pointer==0)
     tempfilepointer = fopen (posfile_name,"rb");
   else
     tempfilepointer = posfile_pointer;

   if (tempfilepointer>0)
   {
      if (pos_file_binary==false)
      {
	 long temprecordsize = getrecordsizetext(tempfilepointer);
	 if (temprecordsize>0)
	 {
	    recordsize=temprecordsize;
	    posfile_pointer=tempfilepointer;
	 }
	 else
	 {
	    posfile_ready=false;
	    fclose (tempfilepointer);
	 }; //recordsize ==0
      }else //binary
      {
	 if (getlastposition(tempfilepointer)/(sizeof(float)*8)>0)
	 {
	    recordsize=sizeof(float)*8;
	    posfile_pointer=tempfilepointer;
	 }
	 else
	 {
	    posfile_ready=false;
	    fclose (tempfilepointer);
	 }; //no data sets in file

      };
   } else
   {
      posfile_ready=false;
   };//tempfilepointer<=0

   return  posfile_ready;
}

void CRigidBody::init()
{
   tobj = gluNewTess();
#ifdef UNIX
   gluTessCallback(tobj, (GLenum) GLU_TESS_VERTEX, (GLvoid (*)()) vertexCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_BEGIN, (GLvoid (*)())   beginCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_END,   (GLvoid (*)()) endCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_ERROR,   (GLvoid (*)()) errorCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_COMBINE,  (GLvoid (*)())  combineCallbackag);
//   gluTessCallback(tobj, (GLenum) GLU_TESS_EDGE_FLAG_DATA, (GLvoid (*)()) edgeFlagData);
#elif defined WINDOWS
   gluTessCallback(tobj, (GLenum) GLU_TESS_VERTEX,(void (__stdcall *)(void))    vertexCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_BEGIN, (void (__stdcall *)(void))    beginCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_END,   (void (__stdcall *)(void))  endCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_ERROR,   (void (__stdcall *)(void))  errorCallbackag);
   gluTessCallback(tobj, (GLenum) GLU_TESS_COMBINE,  (void (__stdcall *)(void))   combineCallbackag);
//   gluTessCallback(tobj, (GLenum) GLU_TESS_EDGE_FLAG_DATA, (void (__stdcall *)(void))   edgeFlagData);
#endif
}

void CRigidBody::ret_time(float* time)
{
   *time=current_time;
}

void CRigidBody::ret_trans(float* translation)
{
   translation[0]=position_x  ;
   translation[1]=position_y ;
   translation[2]=position_z  ;
}

void CRigidBody::ret_rot(float* rotation)
{
   rotation[0]=angle_alpha;
   rotation[1]=angle_beta;
   rotation[2]=angle_gamma;
}

void CRigidBody::return_max_dimensions(float* min_x, float* max_x, float* min_y, float* max_y,float* min_z, float* max_z)
{
   *min_x = x_min;
   *max_x = x_max;
   *min_y = y_min;
   *max_y = y_max;
   *min_z = z_min;
   *max_z = z_max;

}

void CRigidBody::draw_pure()
{
   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   glDisable(GL_LIGHTING);
   glEnable(GL_COLOR_MATERIAL);
   glColor3fv(pick_color_value);
   glPushMatrix();
   glTranslated(position_x, position_y, position_z);
   glMultMatrixf(object_rotation_matrix);
   glTranslated(position_x_initial, position_y_initial, position_z_initial);
   glMultMatrixf(object_initial_matrix);
   glScaled(scale_factor, scale_factor, scale_factor);
   glCallList(Displ_List);
   glPopMatrix();
   glEnable(GL_LIGHTING);
   glDisable(GL_COLOR_MATERIAL);
}

void CRigidBody::return_info_string(QString &info)
{
   char degchar[2]={0xC2, 0xB0};
   QString deg=QString::fromUtf8(degchar,2);
   float alphadeg=angle_alpha*180/M_PI;
   float betadeg=angle_beta*180/M_PI;
   float gammadeg=angle_gamma*180/M_PI;
   info=tr("Class: %1\nBody: %2\nPos.: %3\nx: %4\ny: %5\nz: %6\nAlpha: %7%8 (%9%10)\nBeta: %11%12 (%13%14)\nGamma: %15%16 (%17%18)\nColor: %19").arg(class_name).arg(body_file_name).arg(posfile_name).arg(position_x).arg(position_y).arg(position_z).arg(alphadeg).arg(deg).arg(alphadeg-int(alphadeg/360)*360).arg(deg).arg(betadeg).arg(deg).arg(betadeg-int(betadeg/360)*360).arg(deg).arg(gammadeg).arg(deg).arg(gammadeg-int(gammadeg/360)*360).arg(deg).arg(colorvalue);
}

void CRigidBody::edit() {
  if(editDialog==NULL) {
    int n;
    editDialog=new QDialog;
    editDialog->setWindowTitle(tr("Edit %1").arg(posfile_name));
    QGridLayout *layout=new QGridLayout;
    editDialog->setLayout(layout);

    while(angle_alpha_initial>M_PI) { angle_alpha_initial-=2*M_PI; }
    while(angle_beta_initial>M_PI) { angle_beta_initial-=2*M_PI; }
    while(angle_gamma_initial>M_PI) { angle_gamma_initial-=2*M_PI; }
    while(angle_alpha_initial<-M_PI) { angle_alpha_initial+=2*M_PI; }
    while(angle_beta_initial<-M_PI) { angle_beta_initial+=2*M_PI; }
    while(angle_gamma_initial<-M_PI) { angle_gamma_initial+=2*M_PI; }

    // Label
    QLabel *transL, *transLL, *transRL, *transWL=0, *rotL, *rotKL, *rotLL, *rotRL, *rotWL=0, *rotPoint, *drawCosSub=0;
    transL=new QLabel(tr("Init. trans.:"));
    transLL=new QLabel(tr("Move along\nsystem L:"));
    transRL=new QLabel(tr("Move along\nsystem R:"));
    if(!isSubBody) transWL=new QLabel(tr("Move along\nsystem W:"));
    rotL=new QLabel(tr("Init. rot.:\n(Kardan)"));
    rotKL=new QLabel(tr("Init. rot.:\n(Kardan)"));
    rotLL=new QLabel(tr("Rot. around\naxis of\nsystem L:"));
    rotRL=new QLabel(tr("Rot. around\naxis of\nsystem R:"));
    if(!isSubBody) rotWL=new QLabel(tr("Rot. around\naxis of\nsystem W:"));
    rotPoint=new QLabel(tr("Rot. center:"));
    if(isSubBody) drawCosSub=new QLabel(tr("Draw cos.:"));
    n=0;
    layout->addWidget(transL, n, 0); n++;
    layout->addWidget(transLL, n, 0); n++;
    layout->addWidget(transRL, n, 0); n++;
    if(!isSubBody) { layout->addWidget(transWL, n, 0); n++; }
    layout->addWidget(rotL, n, 0); n++;
    layout->addWidget(rotKL, n, 0); n++;
    layout->addWidget(rotLL, n, 0); n++;
    layout->addWidget(rotRL, n, 0); n++;
    if(!isSubBody) { layout->addWidget(rotWL, n, 0); n++; }
    layout->addWidget(rotPoint, n, 0); n++;
    if(isSubBody) layout->addWidget(drawCosSub, n, 0); n++;

    // SpinBox
    for(int i=0; i<=2; i++) {
      // Degree char
      char degchar[2]={0xC2, 0xB0};
      QString deg=QString::fromUtf8(degchar,2);
      trans[i]=new QDoubleSpinBox;
      rot[i]=new QDoubleSpinBox;
      trans[i]->setRange(-1e2, 1e2);
      rot[i]->setRange(-180, 180);
      trans[i]->setDecimals(5);
      rot[i]->setDecimals(3);
      trans[i]->setSingleStep(0.01);
      rot[i]->setSingleStep(10);
      rot[i]->setSuffix(deg);
      trans[i]->setMaximumWidth(70);
      rot[i]->setMaximumWidth(70);
      layout->addWidget(trans[i],0,i+1);
      n=4; if(isSubBody) n--;
      layout->addWidget(rot[i],n,i+1);
    }
    trans[0]->setPalette(QPalette(QColor(129,0,0)));
    trans[1]->setPalette(QPalette(QColor(0,129,0)));
    trans[2]->setPalette(QPalette(QColor(0,0,129)));
    trans[0]->setValue(position_x_initial);
    trans[1]->setValue(position_y_initial);
    trans[2]->setValue(position_z_initial);
    rot[0]->setValue(angle_alpha_initial*180/M_PI);
    rot[1]->setValue(angle_beta_initial*180/M_PI);
    rot[2]->setValue(angle_gamma_initial*180/M_PI);
    
    // Slider
    QwtValueList knobTics[QwtScaleDiv::NTickTypes];
    knobTics[1].append(-180);
    knobTics[1].append(-90);
    knobTics[1].append(0);
    knobTics[1].append(90);
    knobTics[1].append(180);
    knobTics[0].append(-135);
    knobTics[0].append(-45);
    knobTics[0].append(45);
    knobTics[0].append(135);
    QwtScaleDiv knobScale(-180, 180, knobTics);
    for(int i=0; i<=2; i++) {
      transLW[i]=new QwtWheel(editDialog);
      transRW[i]=new QwtWheel(editDialog);
      if(!isSubBody) transWW[i]=new QwtWheel(editDialog);
      rotK[i]=new QwtKnob(editDialog);
      rotLK[i]=new QwtKnob(editDialog);
      rotRK[i]=new QwtKnob(editDialog);
      if(!isSubBody) rotWK[i]=new QwtKnob(editDialog);
      transLW[i]->setTotalAngle(360*10000);
      transRW[i]->setTotalAngle(360*10000);
      if(!isSubBody) transWW[i]->setTotalAngle(360*10000);
      transLW[i]->setRange(-1e2, 1e2, 0.0001, 10);
      transRW[i]->setRange(-1e2, 1e2, 0.0001, 10);
      if(!isSubBody) transWW[i]->setRange(-1e2, 1e2, 0.0001, 10);
      rotK[i]->setRange(-180, 180, 1, 10);
      rotLK[i]->setRange(-180, 180, 1, 10);
      rotRK[i]->setRange(-180, 180, 1, 10);
      if(!isSubBody) rotWK[i]->setRange(-180, 180, 1, 10);
      rotK[i]->setKnobWidth(35);
      rotLK[i]->setKnobWidth(35);
      rotRK[i]->setKnobWidth(35);
      if(!isSubBody) rotWK[i]->setKnobWidth(35);
      rotK[i]->setScale(knobScale);
      rotLK[i]->setScale(knobScale);
      rotRK[i]->setScale(knobScale);
      if(!isSubBody) rotWK[i]->setScale(knobScale);
      connect(transRW[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
      connect(transLW[i], SIGNAL(sliderPressed()), this, SLOT(sliderTPressed()));
      connect(transLW[i], SIGNAL(sliderReleased()), this, SLOT(sliderTReleased()));
      if(!isSubBody) connect(transWW[i], SIGNAL(sliderPressed()), this, SLOT(sliderTPressed()));
      if(!isSubBody) connect(transWW[i], SIGNAL(sliderReleased()), this, SLOT(sliderTReleased()));
      connect(rotK[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
      connect(rotLK[i], SIGNAL(sliderPressed()), this, SLOT(sliderRPressed()));
      connect(rotLK[i], SIGNAL(sliderReleased()), this, SLOT(sliderRReleased()));
      connect(rotRK[i], SIGNAL(sliderPressed()), this, SLOT(sliderRPressed()));
      connect(rotRK[i], SIGNAL(sliderReleased()), this, SLOT(sliderRReleased()));
      if(!isSubBody) connect(rotWK[i], SIGNAL(sliderPressed()), this, SLOT(sliderRPressed()));
      if(!isSubBody) connect(rotWK[i], SIGNAL(sliderReleased()), this, SLOT(sliderRReleased()));
      n=1;
      layout->addWidget(transLW[i],n,i+1); n++;
      layout->addWidget(transRW[i],n,i+1); n++;
      if(!isSubBody) { layout->addWidget(transWW[i],n,i+1); n++; }
      n++;
      layout->addWidget(rotK[i],n,i+1); n++;
      layout->addWidget(rotLK[i],n,i+1); n++;
      layout->addWidget(rotRK[i],n,i+1); n++;
      if(!isSubBody) { layout->addWidget(rotWK[i],n,i+1); n++; }
    }
    transLW[0]->setPalette(QPalette(QColor(255,0,0)));
    transLW[1]->setPalette(QPalette(QColor(0,255,0)));
    transLW[2]->setPalette(QPalette(QColor(0,0,255)));
    transRW[0]->setPalette(QPalette(QColor(129,0,0)));
    transRW[1]->setPalette(QPalette(QColor(0,129,0)));
    transRW[2]->setPalette(QPalette(QColor(0,0,129)));
    if(!isSubBody) transWW[0]->setPalette(QPalette(QColor(255,129,129)));
    if(!isSubBody) transWW[1]->setPalette(QPalette(QColor(129,255,129)));
    if(!isSubBody) transWW[2]->setPalette(QPalette(QColor(129,129,255)));
    rotLK[0]->setPalette(QPalette(QColor(255,0,0)));
    rotLK[1]->setPalette(QPalette(QColor(0,255,0)));
    rotLK[2]->setPalette(QPalette(QColor(0,0,255)));
    rotRK[0]->setPalette(QPalette(QColor(129,0,0)));
    rotRK[1]->setPalette(QPalette(QColor(0,129,0)));
    rotRK[2]->setPalette(QPalette(QColor(0,0,129)));
    if(!isSubBody) rotWK[0]->setPalette(QPalette(QColor(255,129,129)));
    if(!isSubBody) rotWK[1]->setPalette(QPalette(QColor(129,255,129)));
    if(!isSubBody) rotWK[2]->setPalette(QPalette(QColor(129,129,255)));
    connect(transLW[0], SIGNAL(sliderMoved(double)), this, SLOT(sliderTL0Moved(double)));
    connect(transLW[1], SIGNAL(sliderMoved(double)), this, SLOT(sliderTL1Moved(double)));
    connect(transLW[2], SIGNAL(sliderMoved(double)), this, SLOT(sliderTL2Moved(double)));
    if(!isSubBody) connect(transWW[0], SIGNAL(sliderMoved(double)), this, SLOT(sliderTW0Moved(double)));
    if(!isSubBody) connect(transWW[1], SIGNAL(sliderMoved(double)), this, SLOT(sliderTW1Moved(double)));
    if(!isSubBody) connect(transWW[2], SIGNAL(sliderMoved(double)), this, SLOT(sliderTW2Moved(double)));
    connect(rotLK[0], SIGNAL(sliderMoved(double)), this, SLOT(sliderRL0Moved(double)));
    connect(rotLK[1], SIGNAL(sliderMoved(double)), this, SLOT(sliderRL1Moved(double)));
    connect(rotLK[2], SIGNAL(sliderMoved(double)), this, SLOT(sliderRL2Moved(double)));
    connect(rotRK[0], SIGNAL(sliderMoved(double)), this, SLOT(sliderRR0Moved(double)));
    connect(rotRK[1], SIGNAL(sliderMoved(double)), this, SLOT(sliderRR1Moved(double)));
    connect(rotRK[2], SIGNAL(sliderMoved(double)), this, SLOT(sliderRR2Moved(double)));
    if(!isSubBody) connect(rotWK[0], SIGNAL(sliderMoved(double)), this, SLOT(sliderRW0Moved(double)));
    if(!isSubBody) connect(rotWK[1], SIGNAL(sliderMoved(double)), this, SLOT(sliderRW1Moved(double)));
    if(!isSubBody) connect(rotWK[2], SIGNAL(sliderMoved(double)), this, SLOT(sliderRW2Moved(double)));
    transRW[0]->setValue(position_x_initial);
    transRW[1]->setValue(position_y_initial);
    transRW[2]->setValue(position_z_initial);
    rotK[0]->setValue(angle_alpha_initial*180/M_PI);
    rotK[1]->setValue(angle_beta_initial*180/M_PI);
    rotK[2]->setValue(angle_gamma_initial*180/M_PI);

    // Radio Buttons
    rotPointL=new QRadioButton(tr("Sys. L"));
    rotPointR=new QRadioButton(tr("Sys. R"));
    if(!isSubBody) rotPointW=new QRadioButton(tr("Sys. W"));
    n=9; if(isSubBody) n-=2;
    layout->addWidget(rotPointL,n,1);
    layout->addWidget(rotPointR,n,2);
    if(!isSubBody) layout->addWidget(rotPointW,n,3);
    rotPointL->setChecked(true);

    // Draw subbodies cos
    if(isSubBody) {
      QCheckBox *cos;
      cos=new QCheckBox(tr("Show System L and R"));
      layout->addWidget(cos,8,1,1,3);
      cos->setChecked(false);
      local_cos_flag=false;
      connect(cos, SIGNAL(stateChanged(int)), this, SLOT(showCos(int)));
    }

    // Connect
    for(int i=0; i<=2; i++) {
      connect(transRW[i], SIGNAL(valueChanged(double)), trans[i], SLOT(setValue(double)));
      connect(rotK[i], SIGNAL(valueChanged(double)), rot[i], SLOT(setValue(double)));
      connect(trans[i], SIGNAL(valueChanged(double)), transRW[i], SLOT(setValue(double)));
      connect(rot[i], SIGNAL(valueChanged(double)), rotK[i], SLOT(setValue(double)));
    }

    editDialog->resize(300,450);
  }

  editDialog->show();
  editDialog->raise();
  editDialog->activateWindow();

  emit sigUpdateGL();
}

void CRigidBody::updateEdit() {
  position_x_initial=trans[0]->value();
  position_y_initial=trans[1]->value();
  position_z_initial=trans[2]->value();
  angle_alpha_initial=rot[0]->value()*M_PI/180.0;
  angle_beta_initial=rot[1]->value()*M_PI/180.0;
  angle_gamma_initial=rot[2]->value()*M_PI/180.0;
  createInitialRotationMatrix();
  emit sigUpdateGL();
  writeToConsole();
}

void CRigidBody::sliderTPressed() {
  transm[0]=position_x_initial;
  transm[1]=position_y_initial;
  transm[2]=position_z_initial;
}
void CRigidBody::sliderTReleased() {
  for(int i=0; i<=2; i++) {
    transLW[i]->setValue(0);
    if(!isSubBody) transWW[i]->setValue(0);
  }
}

void CRigidBody::sliderRPressed() {
  memcpy(Am, object_initial_matrix, sizeof(GLfloat)*16);
  transm[0]=position_x_initial;
  transm[1]=position_y_initial;
  transm[2]=position_z_initial;
}
void CRigidBody::sliderRReleased() {
  for(int i=0; i<=2; i++) {
    rotLK[i]->setValue(0);
    rotRK[i]->setValue(0);
    if(!isSubBody) rotWK[i]->setValue(0);
  }
}

void CRigidBody::sliderTL0Moved(double v) {
  sliderTMoved(v, 0);
}
void CRigidBody::sliderTL1Moved(double v) {
  sliderTMoved(v, 1);
}
void CRigidBody::sliderTL2Moved(double v) {
  sliderTMoved(v, 2);
}
void CRigidBody::sliderTW0Moved(double v) {
  sliderTMoved(v, 3);
}
void CRigidBody::sliderTW1Moved(double v) {
  sliderTMoved(v, 4);
}
void CRigidBody::sliderTW2Moved(double v) {
  sliderTMoved(v, 5);
}

void CRigidBody::sliderRL0Moved(double v) {
  sliderRMoved(v, 0);
}
void CRigidBody::sliderRL1Moved(double v) {
  sliderRMoved(v, 1);
}
void CRigidBody::sliderRL2Moved(double v) {
  sliderRMoved(v, 2);
}
void CRigidBody::sliderRR0Moved(double v) {
  sliderRMoved(v, 3);
}
void CRigidBody::sliderRR1Moved(double v) {
  sliderRMoved(v, 4);
}
void CRigidBody::sliderRR2Moved(double v) {
  sliderRMoved(v, 5);
}
void CRigidBody::sliderRW0Moved(double v) {
  sliderRMoved(v, 6);
}
void CRigidBody::sliderRW1Moved(double v) {
  sliderRMoved(v, 7);
}
void CRigidBody::sliderRW2Moved(double v) {
  sliderRMoved(v, 8);
}

void CRigidBody::sliderTMoved(double v, int nr) {
  // Relativ movement vector [x; y; z]
  float x, y, z;
  if(nr==0) {
    x=object_initial_matrix[ 0];y=object_initial_matrix[ 1];z=object_initial_matrix[ 2];
  } else if(nr==1) {
    x=object_initial_matrix[ 4];y=object_initial_matrix[ 5];z=object_initial_matrix[ 6];
  } else if(nr==2) {
    x=object_initial_matrix[ 8];y=object_initial_matrix[ 9];z=object_initial_matrix[10];
  } else if(nr==3) {
    x=object_rotation_matrix[ 0];y=object_rotation_matrix[ 4];z=object_rotation_matrix[ 8];
  } else if(nr==4) {
    x=object_rotation_matrix[ 1];y=object_rotation_matrix[ 5];z=object_rotation_matrix[ 9];
  } else {
    x=object_rotation_matrix[ 2];y=object_rotation_matrix[ 6];z=object_rotation_matrix[10];
  }

  // Calculate new initial postition rB=rB+[x;y;z]*v
  position_x_initial=transm[0]+x*v;
  position_y_initial=transm[1]+y*v;
  position_z_initial=transm[2]+z*v;

  // Set GUI values
  for(int i=0; i<=2; i++)
    disconnect(transRW[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
  trans[0]->setValue(position_x_initial);
  trans[1]->setValue(position_y_initial);
  trans[2]->setValue(position_z_initial);

  emit sigUpdateGL();
  writeToConsole();

  for(int i=0; i<=2; i++)
    connect(transRW[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
}

void CRigidBody::sliderRMoved(double v, int nr) {
  v*=M_PI/180;

  // Calculate rot matrix around [x, y, z] axis by angle v = An
  float An[16];
  float x, y, z, c, s, C, xs, ys, zs, xC, yC, zC, xyC, yzC, zxC;
  if(nr==0) {
    x=1; y=0; z=0;
  } else if(nr==1) {
    x=0; y=1; z=0;
  } else if(nr==2) {
    x=0; y=0; z=1;
  } else if(nr==3) {
    x=Am[0]; y=Am[4]; z=Am[8];
  } else if(nr==4) {
    x=Am[1]; y=Am[5]; z=Am[9];
  } else if(nr==5) {
    x=Am[2]; y=Am[6]; z=Am[10];
  } else if(nr==6) {
    vec n, r; n.e[0]=1; n.e[1]=0; n.e[2]=0; n.e[3]=0;
    r=mmv(mtrans(tomat(Am)),mmv(mtrans(tomat(object_rotation_matrix)),n));
    x=r.e[0]; y=r.e[1]; z=r.e[2];
  } else if(nr==7) {
    vec n, r; n.e[0]=0; n.e[1]=1; n.e[2]=0; n.e[3]=0;
    r=mmv(mtrans(tomat(Am)),mmv(mtrans(tomat(object_rotation_matrix)),n));
    x=r.e[0]; y=r.e[1]; z=r.e[2];
  } else {
    vec n, r; n.e[0]=0; n.e[1]=0; n.e[2]=1; n.e[3]=0;
    r=mmv(mtrans(tomat(Am)),mmv(mtrans(tomat(object_rotation_matrix)),n));
    x=r.e[0]; y=r.e[1]; z=r.e[2];
  }
  c=cos(v); s=sin(v); C=1-c;
  xs=x*s; ys=y*s; zs=z*s;
  xC=x*C; yC=y*C; zC=z*C;
  xyC=x*yC; yzC=y*zC; zxC=z*xC;
  An[ 0]=x*xC+c; An[ 4]=xyC-zs; An[ 8]=zxC+ys; An[12]=0;
  An[ 1]=xyC+zs; An[ 5]=y*yC+c; An[ 9]=yzC-xs; An[13]=0;
  An[ 2]=zxC-ys; An[ 6]=yzC+xs; An[10]=z*zC+c; An[14]=0;
  An[ 3]=     0; An[ 7]=     0; An[11]=     0; An[15]=1;

  // Am*An
  matto(mmm(tomat(Am),tomat(An)),object_initial_matrix);

  // Calculate Kardan angles from new initial rotation matrix ARL
  angle_beta_initial=asin(object_initial_matrix[8]);
  float nenner=cos(angle_beta_initial);
  if(nenner>1e-10) {
    angle_alpha_initial=atan2(-object_initial_matrix[9],object_initial_matrix[10]);
    angle_gamma_initial=atan2(-object_initial_matrix[4],object_initial_matrix[0]);
  } else {
    angle_alpha_initial=0;
    angle_gamma_initial=atan2(object_initial_matrix[1],object_initial_matrix[5]);
  }

  // Correct initial translation vector (rotation point L, R or W)
  if(rotPointL->isChecked()) {
    // nothing to do
  } else if(rotPointR->isChecked()) {
    // rBnew=Am*An*Am'*RrB
    vec r=mmv(tomat(Am),mmv(tomat(An),mmv(mtrans(tomat(Am)),tovec(transm))));
    position_x_initial=r.e[0]; position_y_initial=r.e[1]; position_z_initial=r.e[2];
  } else {
    // rBnew=Am*An*Am'*(object_rotation_matrix'*WrP+transm)-object_rotation_matrix'*WrP
    vec WrP, r; WrP.e[0]=position_x; WrP.e[1]=position_y; WrP.e[2]=position_z; WrP.e[3]=0;
    r=vsv(mmv(tomat(Am),mmv(tomat(An),mmv(mtrans(tomat(Am)),vav(mmv(mtrans(tomat(object_rotation_matrix)),WrP),tovec(transm))))),mmv(mtrans(tomat(object_rotation_matrix)),WrP));
    position_x_initial=r.e[0]; position_y_initial=r.e[1]; position_z_initial=r.e[2];
  }

  // Set GUI values
  for(int i=0; i<=2; i++) {
    disconnect(rotK[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
    disconnect(transRW[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
  }
  rot[0]->setValue(angle_alpha_initial*180/M_PI);
  rot[1]->setValue(angle_beta_initial*180/M_PI);
  rot[2]->setValue(angle_gamma_initial*180/M_PI);
  trans[0]->setValue(position_x_initial);
  trans[1]->setValue(position_y_initial);
  trans[2]->setValue(position_z_initial);

  emit sigUpdateGL();
  writeToConsole();

  for(int i=0; i<=2; i++) {
    connect(rotK[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
    connect(transRW[i], SIGNAL(valueChanged(double)), this, SLOT(updateEdit()));
  }
}

void CRigidBody::writeToConsole() {
  printf("\n");
  printf("Current initial transaltion and rotation for body %s:\n", body_file_name);
  printf("%13.5e %13.5e %13.5e \n", position_x_initial, position_y_initial, position_z_initial);
  printf("%13.5e %13.5e %13.5e \n", angle_alpha_initial, angle_beta_initial, angle_gamma_initial);
}

void CRigidBody::showCos(int state) {
  if(state==Qt::Checked)
    local_cos_flag=true;
  else
    local_cos_flag=false;
  emit sigUpdateGL();
}
