/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "drawobjectcontainer.h" // class's header file
#include <testelasticbody.h>
#include <arrow.h>
#include <path.h>
#include <deflectionline.h>
#include <ketsim_objobject.h>
#include <glwidget.h>

using namespace std;

// class constructor
DrawObjectContainer::DrawObjectContainer()
{
   total_beginning_flag = true;
   current_object = 0;
   object_pick_counter = 1;
   beginning_record = 0;
}

// class destructor
DrawObjectContainer::~DrawObjectContainer()
{
   // insert your code here
}

int DrawObjectContainer::GetCharacterLine(char *pCharacterLineBuffer, int iBufferSize, FILE *pFile)
{
   if (pCharacterLineBuffer == NULL || pFile == NULL) return Error_wrong_params;
   if (iBufferSize < 2) return Error_buffertoosmall;
   int ch, count = 0;
   long pos_old;
   while (true)
   {
      if (count == (iBufferSize-1))
      {
	 pCharacterLineBuffer[count] = '\0'; // Terminate
	 return Error_linetoolong;
      }
      ch = fgetc(pFile);
      if (ch != EOF)
      {
	 if ((ch == CR) || (ch == LF))
	 {
	    pCharacterLineBuffer[count] = '\0'; // Terminate
	    if (count==0)

	       continue; // Get next line
	    else
	    {
	       pos_old=ftell (pFile);
	       ch = fgetc(pFile);
	       if ((ch == CR) || (ch == LF)) {}else fseek (pFile,pos_old,SEEK_SET);
	       break;
	    }; // Get out - return success
	 }
	 else {
	    pCharacterLineBuffer[count] = (char)ch;
	    count++;
	 }
      }
      else {
	 pCharacterLineBuffer[count] = '\0'; // Terminate
	 if (!ferror(pFile)) return EOF;
	 return Error_readerror;
      }
   };
   return count;
}

void DrawObjectContainer::parsenextdataline(FILE * pfilepointer)
{
   char* newline=new char[charlength];
   int ret;
   long oldpos;
   while (1 )

   {
      oldpos=ftell(pfilepointer);
      newline[0]='#';
      ret = GetCharacterLine(newline, charlength, pfilepointer);
      if (ret == EOF) break; // Get out of while loop
      if( (newline[0]!='#') ) break;
   };
   fseek (pfilepointer,oldpos,SEEK_SET);

}

CBody* DrawObjectContainer::objectFactory(char *ObjectClassName, char *BodyFileName, char *NewFileEnding) {
  CBody *newobject=0;

  if(strcmp (ObjectClassName,"Cube")==0)                  newobject = new Cube( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Cuboid")==0)                newobject = new Cuboid ( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Torus")==0)                 newobject = new Torus( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Cylinder")==0)              newobject = new Cylinder( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Sphere")==0)                newobject = new Sphere( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"InvisibleBody")==0)         newobject = new InvisibleBody( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"TestElasticBody")==0)       newobject = new TestElasticBody( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Arrow")==0)       	  newobject = new Arrow( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"CoilSpring")==0)       	  newobject = new CoilSpring( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Path")==0)      		  newobject = new Path( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"DeflectionLine")==0)        newobject = new DeflectionLine( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Kos")==0)                   newobject = new Kos( BodyFileName, NewFileEnding);
   // Roland Zander, Thorsten Schindler Start
 // nur zur Rueckwaerts-Kompatibilitaet
  if(strcmp (ObjectClassName,"ElasticBody1sRCM")==0)        newobject = new ElasticBody1s21RCM( BodyFileName, NewFileEnding);
 // -----------------------------------
  if(strcmp (ObjectClassName,"ElasticBody1s21RCM")==0)      newobject = new ElasticBody1s21RCM( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"ElasticBody1s33RCM")==0)      newobject = new ElasticBody1s33RCM( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"ElasticBody1sBTA")==0)        newobject = new ElasticBody1sBTA( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"ElasticBody2s13Disk")==0)     newobject = new ElasticBody2s13Disk( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Quad")==0)                    newobject = new Quad( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Area")==0)                    newobject = new Area( BodyFileName, NewFileEnding);
   // Roland Zander End
  if(strcmp (ObjectClassName,"ObjObject")==0)             newobject = new ObjObject( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Extrusion")==0)             newobject = new Extrusion( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"Rotation")==0)              newobject = new Rotation( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_VZ")==0)             newobject = new KETSIM_VZ(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_FS")==0)             newobject = new KETSIM_FS(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_SS")==0)             newobject = new KETSIM_SS(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_RK_HK_BB")==0)       newobject = new KETSIM_RK_HK_BB( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_RK_HK_NN")==0)       newobject = new KETSIM_RK_HK_NN(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_ZK_AG_FL1")==0)      newobject = new KETSIM_ZK_AG_FL1(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_ZK_AG_FL2")==0)      newobject = new KETSIM_ZK_AG_FL2(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_ZK_ZL")==0)          newobject = new KETSIM_ZK_ZL(BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"KETSIM_ObjObject")==0)      newobject = new KETSIM_ObjObject(BodyFileName, NewFileEnding);
  // Start: VT (valve train) Komponenten (Huber / Schneider)
  if(strcmp (ObjectClassName,"VT_ValveSeat")==0)                 newobject = new VT_ValveSeat( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"VT_Valve")==0)                newobject = new VT_Valve( BodyFileName, NewFileEnding);
  if(strcmp (ObjectClassName,"VT_RockerArm")==0)             newobject = new VT_RockerArm( BodyFileName, NewFileEnding);
  // Ende : VT 
  if(strcmp (ObjectClassName,"CompoundPrimitiveBody")==0) newobject = new CompoundPrimitiveBody(BodyFileName, NewFileEnding);
  
  if (newobject == 0) {
      cout << " No body " << ObjectClassName << " defined." << endl;
  };

  connect(newobject, SIGNAL(sigUpdateGL()), glWidget, SLOT(updateGL()));
  
  return newobject;
}

void DrawObjectContainer::Create()
{
   int bitB, bitG, bitR;
   glGetIntegerv(GL_BLUE_BITS, &bitB);
   glGetIntegerv(GL_GREEN_BITS, &bitG);
   glGetIntegerv(GL_RED_BITS, &bitR);

   QStringList filters;
   filters << "*.body";
   QDir dir;
   bool retry;
   int ret;
   FILE * pFile;
   char* testchar = new char [charlength];
   char* ObjectClassName = new char [charlength];
   char*  NewFileName= new char [charlength];
   char*  NewFileEnding= new char [charlength];
   char*  BodyFileName= new char [charlength];
do {
   retry=false;


   int   NumberOfObjects;
   QString FileNametoopen;
   QString PosDataFileName;
   CBody *newobject;


   dir.setFilter(QDir::Files);
   dir.setNameFilters(filters);
   dir.setSorting(QDir::Name);
   QFileInfoList list = dir.entryInfoList();


if(list.size()>0)
   {
      for (int i = 0; i < list.size(); ++i)
      {
	 QFileInfo fileInfo = list.at(i);
	 FileNametoopen=fileInfo.fileName();
	 pFile = fopen (FileNametoopen.toAscii(),"rb");
	 //Jump to Class Name
	 parsenextdataline(pFile);
	 //read the class name
	 ret = GetCharacterLine(testchar, charlength,pFile);
	 sscanf(testchar, "%s", ObjectClassName);
	 //jump to number of objects
	 parsenextdataline(pFile);
	 ret = GetCharacterLine(testchar, charlength,pFile);
	 sscanf(testchar, "%i\n", &NumberOfObjects);

         for (int j = 1; j <= NumberOfObjects; ++j)
	 {
            PosDataFileName=fileInfo.completeBaseName();
	    //copy the name to the NewFileName variable
	    strcpy(NewFileName,PosDataFileName.toAscii());
	    strcpy(BodyFileName,FileNametoopen.toAscii());
	    //create a new file name by adding the number of copies and the ending pos
	    sprintf(NewFileEnding, "%s.%04i.pos",NewFileName, j);
            newobject=objectFactory(ObjectClassName, BodyFileName, NewFileEnding);
            fclose(newobject->datafile);
	    //   newobject=new ElasticBody1sRCM( BodyFileName, NewFileEnding);

	    newobject->contour_flag =true;
	    newobject->path_flag =false;
            
            newobject->pick_color_value[B]=(object_pick_counter>>0         & (1<<bitB)-1) / float((1<<bitB)-1);
            newobject->pick_color_value[G]=(object_pick_counter>>bitB      & (1<<bitG)-1) / float((1<<bitG)-1);
            newobject->pick_color_value[R]=(object_pick_counter>>bitB+bitG & (1<<bitR)-1) / float((1<<bitR)-1);
               
             //End (changes)
	    object_pick_counter++;
	    newobject->init();
	    OBjectVectorContainer.push_back(newobject);
	    LeftInfoList->addItem(NewFileEnding);

	 }
	 fclose (pFile);   //close the current body file
      };		   //all files found in directory
      // read enviornment bodies
      QStringList filters;
      filters<<"*.bodyenv";
      QDir dir;
      dir.setFilter(QDir::Files);
      dir.setNameFilters(filters);
      dir.setSorting(QDir::Name);
      QFileInfoList list=dir.entryInfoList();
      for(int i=0; i<list.size(); ++i) {
        QFileInfo fileInfo=list.at(i);
	FileNametoopen=fileInfo.fileName();
        FILE *pFile=fopen(FileNametoopen.toAscii(),"rb");
	parsenextdataline(pFile);
	GetCharacterLine(testchar, charlength,pFile);
 	sscanf(testchar, "%s", ObjectClassName);
	strcpy(BodyFileName,FileNametoopen.toAscii());
        CBody *envbody=objectFactory(ObjectClassName, BodyFileName, "");
        fclose(envbody->datafile);
        BodyEnv.push_back(envbody);
	fclose(pFile);
      }
   }else		   //list.size()>0
   {
      retry=true;
      int ret=QMessageBox::question(glWidget, "AMVis",tr("No valid body-files found in directory:\n%1\n\nSelect another directory or abort.").arg(QDir::currentPath()),QMessageBox::Ok,QMessageBox::Abort);
      if(ret!=QMessageBox::Ok)
        exit(1);
      QString newworkdir=QFileDialog::getExistingDirectory(glWidget, tr("Directory of Body/Pos-Files"), QDir::currentPath());
      QDir::setCurrent(newworkdir);
   };

}
while(retry==true);
}

void DrawObjectContainer::draw_all_objects()
{ CBody* TempDrawObject;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
       OBjectVectorContainerIterator != OBjectVectorContainer.end();
       OBjectVectorContainerIterator++)
   {
      TempDrawObject = *OBjectVectorContainerIterator;
      if(TempDrawObject->enabled == true) TempDrawObject->draw();

   }
}


void DrawObjectContainer::draw_all_objects_pure()
{ CBody* TempDrawObject;

   vector <CBody*>::iterator OBjectVectorContainerIterator;
   for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
       OBjectVectorContainerIterator != OBjectVectorContainer.end();
       OBjectVectorContainerIterator++)
   {

      TempDrawObject = *OBjectVectorContainerIterator;
      if(TempDrawObject->enabled == true) TempDrawObject->draw_pure();

   }
}


long DrawObjectContainer::return_picked_object( float *picked_color)
{
   CBody* TempDrawObject;
   long position =0;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
       OBjectVectorContainerIterator != OBjectVectorContainer.end();
       OBjectVectorContainerIterator++)
   {

      TempDrawObject = *OBjectVectorContainerIterator;

      if(fabs(picked_color[B]-TempDrawObject->pick_color_value[B])<1e-5 &&
         fabs(picked_color[G]-TempDrawObject->pick_color_value[G])<1e-5 &&
         fabs(picked_color[R]-TempDrawObject->pick_color_value[R])<1e-5)
        return position;

      position++;
   }
   return -1;
}

long DrawObjectContainer::update_last_position()
{
   CBody* TempDrawObject;
   long lastentry;

   vector <CBody*>::iterator OBjectVectorContainerIterator;

   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   lastentry = TempDrawObject->get_dataset_last();
   if (lastentry>1)
   {
      for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
	  OBjectVectorContainerIterator != OBjectVectorContainer.end();
	  OBjectVectorContainerIterator++)
      {

	 TempDrawObject = *OBjectVectorContainerIterator;
	 TempDrawObject->update(lastentry-2);

      }
   }
   return lastentry-1;
}


long DrawObjectContainer::update_current_position()
{
   CBody* TempDrawObject;
   long current_entry ,lastentry;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   if (total_beginning_flag ==true)
   {
      current_entry = 0;
      total_beginning_flag = false;
   }else
   {
      current_entry = TempDrawObject->get_dataset();
   };
   lastentry = TempDrawObject->get_dataset_last();
   if (current_entry<lastentry-1)
   {
      for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
	  OBjectVectorContainerIterator != OBjectVectorContainer.end();
	  OBjectVectorContainerIterator++)
      {
      	 TempDrawObject = *OBjectVectorContainerIterator;
	     TempDrawObject->update(current_entry);
      }
   }
   return current_entry;
}


float DrawObjectContainer::object_time()
{
   CBody* TempDrawObject;

   float time;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   TempDrawObject->ret_time(&time);
   return time;
}


long  DrawObjectContainer::get_max_records()
{
   CBody* TempDrawObject;
   long MaxRec;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   MaxRec = TempDrawObject->get_dataset_last();
   return MaxRec;
}

void DrawObjectContainer::set_update_position(long position)
{
   CBody* TempDrawObject;
   long lastentry;

   vector <CBody*>::iterator OBjectVectorContainerIterator;
   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   lastentry = TempDrawObject->get_dataset_last();
   if (position<lastentry-1)
   {
      for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
	  OBjectVectorContainerIterator != OBjectVectorContainer.end();
	  OBjectVectorContainerIterator++)
      {
      	 TempDrawObject = *OBjectVectorContainerIterator;
	    TempDrawObject->update(position);
      }
   }
   else
   {
        for(OBjectVectorContainerIterator = OBjectVectorContainer.begin();
	  OBjectVectorContainerIterator != OBjectVectorContainer.end();
	  OBjectVectorContainerIterator++)
      {
      	 TempDrawObject = *OBjectVectorContainerIterator;
	    TempDrawObject->update(lastentry-2);
      }   
   }
}

void DrawObjectContainer::get_current_trans(float* return_trans)
{
   CBody* TempDrawObject;
   TempDrawObject = OBjectVectorContainer[current_object];
   TempDrawObject->ret_trans(return_trans);


}


void DrawObjectContainer::get_delta_time(float* deltatime)
{
  
   CBody* TempDrawObject;
   float time1, time2;
   long current_dataset;
   long last_dataset;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   
   last_dataset = TempDrawObject->get_dataset_last();

   if (last_dataset>1)
   {
   current_dataset = TempDrawObject->get_dataset();
   TempDrawObject->update(0);
   TempDrawObject->ret_time(&time1);
   TempDrawObject->update(1);
   TempDrawObject->ret_time(&time2); 
   TempDrawObject->update(current_dataset-1);
   *deltatime = time2 - time1;
   }
   else
   *deltatime =  0.0;
}

long DrawObjectContainer::get_current_record()
{
   CBody* TempDrawObject;
   long dataset;
   vector <CBody*>::iterator OBjectVectorContainerIterator;
   OBjectVectorContainerIterator = OBjectVectorContainer.begin();
   TempDrawObject = *OBjectVectorContainerIterator;
   dataset = TempDrawObject->get_dataset();
   return dataset;
}


void DrawObjectContainer::get_current_rot(float* return_rot)
{
   CBody* TempDrawObject;
   TempDrawObject = OBjectVectorContainer[current_object];
   TempDrawObject->ret_rot(return_rot);
}

void DrawObjectContainer::get_object_info(QString &ob_info)
{
   CBody* TempDrawObject;
   QString Info;
   Info.setNum(OBjectVectorContainer.size());
   TempDrawObject = OBjectVectorContainer[current_object];
   TempDrawObject->return_info_string(ob_info);
}

void DrawObjectContainer::set_current_object(int objectnumber)
{
   current_object=	objectnumber;
}


void DrawObjectContainer::get_object_properties(bool* ob_enabled, bool* ob_selected, bool* ob_contour, bool *ob_path, bool* ob_local_cos, int *mode_)
{
   CBody* TempDrawObject;
   QString Info;
   Info.setNum(OBjectVectorContainer.size());
   TempDrawObject = OBjectVectorContainer[current_object];
   *ob_enabled = TempDrawObject->enabled;
   *ob_selected = TempDrawObject->selected_flag;
   *ob_contour = TempDrawObject->contour_flag;
   *ob_path = TempDrawObject->path_flag;
   *ob_local_cos = TempDrawObject->local_cos_flag;
   *mode_=TempDrawObject->mode;
}

void DrawObjectContainer::set_object_properties(bool ob_enabled, bool ob_selected, bool ob_contour,bool ob_path,bool ob_local_cos)
{
   CBody* TempDrawObject;
   QString Info;
   Info.setNum(OBjectVectorContainer.size());
   TempDrawObject = OBjectVectorContainer[current_object];
   TempDrawObject->enabled = ob_enabled;
   TempDrawObject->selected_flag = ob_selected;
   TempDrawObject->contour_flag = ob_contour ;
   TempDrawObject->path_flag = ob_path ;
   TempDrawObject->local_cos_flag = ob_local_cos;
}

void DrawObjectContainer::set_draw_mode(int drawmode)
{
   CBody* TempDrawObject;
   TempDrawObject = OBjectVectorContainer[current_object];
   TempDrawObject->mode=drawmode;
}


void DrawObjectContainer::get_object_dimensions(float* minimum_x, float* maximum_x, float* minimum_y, float* maximum_y,float* minimum_z, float* maximum_z)
{
   CBody* TempDrawObject;
   QString Info;
   Info.setNum(OBjectVectorContainer.size());
   TempDrawObject = OBjectVectorContainer[current_object];
   TempDrawObject->return_max_dimensions( minimum_x,  maximum_x,  minimum_y,  maximum_y, minimum_z,  maximum_z);

}

int  DrawObjectContainer::get_current_object()
{
   return current_object;
}


void DrawObjectContainer::editBody() {
  int row=LeftInfoList->currentRow();
  CBody *body;
  body=OBjectVectorContainer[row];
  body->edit();
}
