/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <elastic.h>

using namespace std;

// Constructor
ElasticBody::ElasticBody(char* body_name, char *pos_file) : CBody(body_name, pos_file) {
    recordsize = 0;
    posfile_pointer = NULL;
}
// Destructor
ElasticBody::~ElasticBody() {
}

// Initialize (Call CBody::init()!)
void ElasticBody::init() {
    CBody::init();
}

void ElasticBody::draw() {
    if(mode==0) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    if(mode==1) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    if(mode==2) glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);
    
    // Licht
    glEnable(GL_LIGHTING);
    glEnable(GL_BLEND);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_SMOOTH);
    glEnable (GL_NORMALIZE);
    glShadeModel(GL_SMOOTH);
}

// For move camera with body (rigid-body movement)
void ElasticBody::ret_rot(float *rotation) {
  rotation[0]=0;
  rotation[1]=0;
  rotation[2]=0;
}

// For move camera with body (rigid-body movement)
void ElasticBody::ret_trans(float *translation) {
  translation[0]=0;
  translation[1]=0;
  translation[2]=0;
}

// Read data from pos-file
void ElasticBody::update(long dataset_number) {
  currentFrame = dataset_number;
  fseek(posfile_pointer, dataset_number*recordsize, SEEK_SET);
  if(pos_file_binary==true) {
      float dummy[dofs+1];
      fread(dummy,sizeof(float),(dofs+1),posfile_pointer);
      current_time = dummy[0];
      for(int i=0;i<dofs;i++)
	  q[i] = dummy[i+1];
  } else {
      fscanf(posfile_pointer, "%f", &current_time);
      for(int i=0;i<dofs;i++)
	  fscanf(posfile_pointer, "%f", &q[i]);
  } 
  curdataset=dataset_number;
}

// Text in the left
void ElasticBody::return_info_string(QString &info) {
  info=tr("Class: %1\nBody: %2").arg(class_name).arg(body_file_name);
}

// Constructor
ElasticBody1s::ElasticBody1s(char* body_name, char *pos_file) : ElasticBody(body_name, pos_file), discretisation(64) {
    int result;
    char* charstringline = new char [charlength];

    // Farbe
    parsenextdataline(datafile);
    result=GetCharacterLine(charstringline, charlength, datafile);
    sscanf(charstringline, "%f" ,&colorvalue);
    if(0.0 <= colorvalue && colorvalue <= 1.0)  translateColorvalueToRGBA(colorvalue, &color[0], &color[1], &color[2], &color[3]);
    else {
      color[0] = 0.65;
      color[1] = 0.65;
      color[2] = 1.25;
      color[3] = 1.25;
    }
}

// Destructor
ElasticBody1s::~ElasticBody1s() {
}

// draw
void ElasticBody1s::draw() {
    ElasticBody::draw();
    
    // Farben
    glEnable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
    glColorMaterial (GL_FRONT_AND_BACK, GL_DIFFUSE);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
    static GLfloat mat_specular[] = {1.0,1.0,1.0, 1.0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
    static GLfloat color_shininess = 20;
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
    float em[]={0,0,0,0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);

    // TUBE_NORM_FACET //-> kantig
    // TUBE_NORM_PATH_EDGE  //-> glatt
    gleSetJoinStyle ( TUBE_NORM_PATH_EDGE | TUBE_JN_ANGLE | TUBE_JN_CAP);

    glEnable(GL_LINE_SMOOTH);
    drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!
    glDisable(GL_LINE_SMOOTH);

    if(contour_flag==true) {
	// draw contour
//    gleTwistExtrusion( nContourPoints, contour, cont_normal, up, discretisation, point_array, NULL, twist_array );
	// or
//	drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!
    }
    
    if(selected_flag==true) {
	// draw something
    }
    
    if(local_cos_flag==true) {
	// draw something
    }
    
    glEnable(GL_COLOR_MATERIAL);
}

void ElasticBody1s::edit() {
  cout << "plot neutral fiber for body " << posfile_name << endl;
  cout << "current frame = " << currentFrame << endl;
  char filename[128];
  sprintf(filename,"frame%d.%s",currentFrame,posfile_name);
  cout << "file-name : " << filename << endl;

  FILE *cframeFile = fopen(filename,"w");

  gleDouble point_array[discretisation+2][3];
  gleDouble twist_array[discretisation+2];
  computeSweep(discretisation,point_array,twist_array);

  fprintf(cframeFile,"# x\ty\tz\talpha\n");
  for(int i=0;i<discretisation+2;i++)
    fprintf(cframeFile,"%f\t%f\t%f\t%f\n",point_array[i][0],point_array[i][1],point_array[i][2],twist_array[i]);

  fclose(cframeFile);
}

// For selection by mouse click
void ElasticBody1s::draw_pure() {
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_LIGHTING);
  glEnable(GL_COLOR_MATERIAL);
  glColor3fv(pick_color_value);
  glPushMatrix();

  drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!

  glPopMatrix();
  glEnable(GL_LIGHTING);
  glDisable(GL_COLOR_MATERIAL);
}

// die eigentliche Zeichnerei...
void ElasticBody1s::drawBody() {
  gleDouble point_array[discretisation+2][3];
  gleDouble twist_array[discretisation+2];
  computeSweep(discretisation,point_array,twist_array);
  
  gleTwistExtrusion( nContourPoints, contour, cont_normal, up, discretisation+2, point_array, NULL, twist_array ); // color=NULL -> used predefined color
}


