/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <elastic1s21rcm.h>

using namespace std;

// Constructor
ElasticBody1s21RCM::ElasticBody1s21RCM(char* body_name, char *pos_file) : ElasticBody1s(body_name, pos_file) {
  int result;
  char* charstringline = new char [charlength];

/*
  Parameter im File: ---------------
  # Class Name
  ElasticBody1s21RCM
  
  # Instances
  1
  
  # ASCI (0) or Binary (1)
  0
# Color
0  
  # Class Stuff
  2
  o
  0.25
  0.0, 0.0, 0.0
  ----------------------------------
 */
  
  // Klassen-Auslesen beginnt bei Class Stuff

  // Zahl der Elemente
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%i" ,&nElements);

  // Offene oder Ringstruktur
  char cclosed;
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%c" ,&cclosed);
  if (cclosed == 'c') closed = true;
  else                closed = false;

  if(closed) {
      dofs = ( 5*nElements     );
      binaryrecordsize=sizeof(float)*(dofs + 1); // Zeit nicht vergessen
      q = new float[dofs];
  }
  else       {
      dofs = ( 5*nElements + 3 );
      binaryrecordsize=sizeof(float)*(dofs + 1); // Zeit nicht vergessen
      q = new float[dofs];
  }

  // Elementlaenge
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&l0);

  lGes    = nElements*l0;
  lOffSet = lGes * 0.5e-5;
  lGesEff = lGes - 2*lOffSet;

  // WrN00
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&WrN00[0],&WrN00[1],&WrN00[2]);

  // JT
  parsenextdataline(datafile); // erste Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f" ,&JT[0][0],&JT[0][1]);
  parsenextdataline(datafile); // zweite Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f" ,&JT[1][0],&JT[1][1]);
  parsenextdataline(datafile); // dritte Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f" ,&JT[2][0],&JT[2][1]);
  // JR
  parsenextdataline(datafile); // 
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&JR[0],&JR[1],&JR[2]);

  l0h2 = l0  *l0;
  l0h4 = l0h2*l0h2;

  // Cylinder (see http://pyopengl.sourceforge.net/documentation/manual/index.xml)
  float cylinderRadius, cuboidHeight;
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline,charlength,datafile);
  sscanf(charstringline, "%f %f" ,&cylinderRadius,&cuboidHeight);
  if (cylinderRadius > 0. && cuboidHeight == -1.) {
    nContourPoints = 33;

    cont_normal = new SVec[nContourPoints];
    contour     = new SVec[nContourPoints];

    for(int i=0; i<nContourPoints; i++) {
      cont_normal[i][0] = cos(2*M_PI/(nContourPoints-1)*i);
      cont_normal[i][1] = sin(2*M_PI/(nContourPoints-1)*i);
      contour[i][0]     = cylinderRadius*cont_normal[i][0];
      contour[i][1]     = cylinderRadius*cont_normal[i][1];
    }
    up[1] = 1.0;
  }
  else if (cylinderRadius>0. && cuboidHeight>0.) {
    float breadth = cylinderRadius;
    float height  = cuboidHeight;

    nContourPoints = 8; // 4 corners of rectangular belong to two sides, respectively
    cont_normal = new SVec[nContourPoints]; // (y,z)
    contour     = new SVec[nContourPoints]; // (y,z)

    contour[0][0] =  0.5*breadth;  cont_normal[0][0]     =  0.0;  
    contour[0][1] =  0.5*height;  cont_normal[0][1]     =  1.0;

    contour[1][0] = -0.5*breadth;  cont_normal[1][0]     =  0.0; 
    contour[1][1] =  0.5*height;  cont_normal[1][1]     =  1.0;

    contour[2][0] = -0.5*breadth;  cont_normal[2][0]     = -1.0; 
    contour[2][1] =  0.5*height;  cont_normal[2][1]     =  0.0;

    contour[3][0] = -0.5*breadth;  cont_normal[3][0]     = -1.0; 
    contour[3][1] = -0.5*height;  cont_normal[3][1]     =  0.0;

    contour[4][0] = -0.5*breadth;  cont_normal[4][0]     =  0.0; 
    contour[4][1] = -0.5*height;  cont_normal[4][1]     = -1.0;

    contour[5][0] =  0.5*breadth;  cont_normal[5][0]     =  0.0; 
    contour[5][1] = -0.5*height;  cont_normal[5][1]     = -1.0;

    contour[6][0] =  0.5*breadth;  cont_normal[6][0]     =  1.0; 
    contour[6][1] = -0.5*height;  cont_normal[6][1]     =  0.0;

    contour[7][0] =  0.5*breadth;  cont_normal[7][0]     =  1.0; 
    contour[7][1] =  0.5*height;  cont_normal[7][1]     =  0.0;
  }
  else {
    nContourPoints = 33;

    cont_normal = new SVec[nContourPoints];
    contour     = new SVec[nContourPoints];

    for(int i=0; i<nContourPoints; i++) {
      cont_normal[i][0] = cos(2*M_PI/(nContourPoints-1)*i);
      cont_normal[i][1] = sin(2*M_PI/(nContourPoints-1)*i);
      contour[i][0]     = 0.001*cont_normal[i][0];
      contour[i][1]     = 0.001*cont_normal[i][1];
    }
  }
  //  // Breite, nur zur Darstellung
  //  breite = l0/30.;
  //
  //  nContourPoints = 8;
  //  cont_normal = new SVec[nContourPoints];
  //  contour     = new SVec[nContourPoints];
  //  
  //  contour[0][0] =  0.5*breite;  cont_normal[0][0]     =  0.0;
  //  contour[0][1] =  0.5*breite;  cont_normal[0][1]     =  1.0;
  //
  //  contour[1][0] = -0.5*breite;  cont_normal[1][0]     =  0.0;
  //  contour[1][1] =  0.5*breite;  cont_normal[1][1]     =  1.0;
  ////
  //  contour[2][0] = -0.5*breite;  cont_normal[2][0]     = -1.0;
  //  contour[2][1] =  0.5*breite;  cont_normal[2][1]     =  0.0;
  //
  //  contour[3][0] = -0.5*breite;  cont_normal[3][0]     = -1.0;
  //  contour[3][1] = -0.5*breite;  cont_normal[3][1]     =  0.0;
  ////
  //  contour[4][0] = -0.5*breite;  cont_normal[4][0]     =  0.0;
  //  contour[4][1] = -0.5*breite;  cont_normal[4][1]     = -1.0;
  //
  //  contour[5][0] =  0.5*breite;  cont_normal[5][0]     =  0.0;
  //  contour[5][1] = -0.5*breite;  cont_normal[5][1]     = -1.0;
  ////
  //  contour[6][0] =  0.5*breite;  cont_normal[6][0]     =  1.0;
  //  contour[6][1] = -0.5*breite;  cont_normal[6][1]     =  0.0;
  //
  //  contour[7][0] =  0.5*breite;  cont_normal[7][0]     =  1.0;
  //  contour[7][1] =  0.5*breite;  cont_normal[7][1]     =  0.0;
  //
  up[1] = 1.0;

}

// Destructor
ElasticBody1s21RCM::~ElasticBody1s21RCM() {
}

void ElasticBody1s21RCM::computeSweep(const int npoints, gleDouble point_array[][3], gleDouble twist_array[]) {
    double ds = lGesEff/(npoints-1);
    
    GLdouble XTemp[3];
    
    // startpoint, not drawn
    LocateStructure(XTemp,0.); 
    point_array[0][0] = XTemp[0];
    point_array[0][1] = XTemp[1];
    point_array[0][2] = XTemp[2];
    twist_array[0]    = 0.0;

    for(int i = 0; i < npoints ; i++) {
 	LocateStructure(XTemp,lOffSet + ds*i); 
	point_array[i+1][0] = XTemp[0];
	point_array[i+1][1] = XTemp[1];
	point_array[i+1][2] = XTemp[2];
	twist_array[i+1]    = 0.0;
    }

    // endpoint, not drawn
    LocateStructure(XTemp,lGes); 
    point_array[npoints+1][0] = XTemp[0];
    point_array[npoints+1][1] = XTemp[1];
    point_array[npoints+1][2] = XTemp[2];
    twist_array[npoints+1]    = 0.0;
}


// Initialize (Call ElasticBody1s::init()!)
void ElasticBody1s21RCM::init() {
	initCoordinates();
    ElasticBody1s::init();
    //initCoordinates();
}


// For move camera with body (rigid-body movement)
void ElasticBody1s21RCM::ret_rot(float *rotation) {
    rotation[0] = 0.0;
    rotation[1] = 0.0;
    rotation[2] = 0.0;
}



// For move camera with body (rigid-body movement)
void ElasticBody1s21RCM::ret_trans(float *translation) {
    translation[0] = JT[0][0]*q[0] + JT[0][1]*q[1] + WrN00[0];
    translation[1] = JT[1][0]*q[0] + JT[1][1]*q[1] + WrN00[1];
    translation[2] = JT[2][0]*q[0] + JT[2][1]*q[1] + WrN00[2];
}



// Text in the left
void ElasticBody1s21RCM::return_info_string(QString &info) {
    if(closed) {
	info=tr("Class: %1\nBody: %2\nPos-File: %3\nRing\nElemets: %4\nl_0: %5\n%6,%7,%8").arg(class_name).arg(body_file_name).arg(posfile_name).arg(nElements).arg(l0).arg(WrN00[0]).arg(WrN00[1]).arg(WrN00[2]);
    }
    else {
	info=tr("Class: %1\nBody: %2\nPos-File: %3\nBeam\nElemets: %4\nl_0: %5\n%6,%7,%8").arg(class_name).arg(body_file_name).arg(posfile_name).arg(nElements).arg(l0).arg(WrN00[0]).arg(WrN00[1]).arg(WrN00[2]);
    }
}







//---------------------------------------------------------------------------
void ElasticBody1s21RCM::LocateStructure(GLdouble *X, const double s)
{
    // das richtige Element ausw�hlen ...
    int    nElement =      0;
    double sElement = -l0/2.;

    while( nElement*l0 <= s)
	nElement++;
    nElement--; // Wir sind eins zu weit gerannt...

    sElement = s - ( nElement + 0.5 ) * l0; // ElementKOS in der Elementmitte gilt s==0

    if     (nElements <= nElement && closed ) nElement -= nElements; // Ringschluss !!!
    else if(nElements <= nElement           ) {
	nElement  -= 1;
	sElement +=l0;
    }

    // die richtigen Daten einfiltern
    BuildElement(nElement);
    LocateBalken(X,sElement);
}

//---------------------------------------------------------------------------
void ElasticBody1s21RCM::BuildElement(const int& ENumber)
{
  int n = 5 * ENumber ;

  for(int i=0;i<8;i++) {
	if(n+i < dofs) qElement[i] = q[n+i]; // hiermit auch Unterscheidung Ring/Kragbalken!!!
	else           qElement[i] = q[i-5];
  }

  if(closed && ENumber == nElements-1)
  { 
	if(qElement[2] - q[2] > 0.0)
 	  qElement[7] += 2*M_PI;
	else
	  qElement[7] -= 2*M_PI;
  }
}

//---------------------------------------------------------------------------
void ElasticBody1s21RCM::LocateBalken(GLdouble *X, const double s)
{
    GLdouble XLocal[2];

    // globale DOFs
    double x1   = qElement[0];    double y1  = qElement[1];    
    double phi1 = qElement[2];
    double a1   = qElement[3];    double a2  = qElement[4];
    double x2   = qElement[5];    double y2  = qElement[6];
    double phi2 = qElement[7];

    // Rechenaufwand senken durch "Zentralisierung"
    double one_p_cos_dphi = (1 + cos(phi1 - phi2));
	
    // interne DOFs -- lokale Koordinate
	double xS   = (36*(x1 + x2)*cos(phi1 - phi2) + 36*(x1 + x2 + (y1 - y2)*sin(phi1 - phi2)) - (64*a1 + 64*a2 - 5*l0*phi1 + 5*l0*phi2)*(sin(phi1) + sin(phi2)))/(72.*one_p_cos_dphi);
	double yS   = ((64*a1 + 64*a2 + 5*l0*(-phi1 + phi2))*(cos(phi1) + cos(phi2)) + 36*(y1 + y2 + (y1 + y2)*cos(phi1 - phi2) + (-x1 + x2)*sin(phi1 - phi2)))/(72.*one_p_cos_dphi);
	double phiS = (2*(-8*a1 + 8*a2 + l0*(phi1 + phi2)) + 2*(-8*a1 + 8*a2 + l0*(phi1 + phi2))*cos(phi1 - phi2) - 11*(y1 - y2)*(cos(phi1) + cos(phi2)) + 11*(x1 - x2)*(sin(phi1) + sin(phi2)))/(4.*l0*one_p_cos_dphi);
	double eps  = -(36*l0 + 36*l0*cos(phi1 - phi2) + 36*(x1 - x2)*(cos(phi1) + cos(phi2)) - (64*a1 + 64*a2 + 5*l0*(-phi1 + phi2))*sin(phi1 - phi2) + 36*(y1 - y2)*(sin(phi1) + sin(phi2)))/(36.*l0*one_p_cos_dphi);
	double aL   = (64*a1 + 64*a2 - 5*l0*phi1 + 5*l0*phi2 + (64*a1 + 64*a2 + 5*l0*(-phi1 + phi2))*cos(phi1 - phi2) - 36*(y1 - y2)*(cos(phi1) + cos(phi2)) + 36*(x1 - x2)*(sin(phi1) + sin(phi2)))/(72.*one_p_cos_dphi);
	double bL   = (2*(-8*a1 + 8*a2 + l0*(-phi1 + phi2)) + 2*(-8*a1 + 8*a2 + l0*(-phi1 + phi2))*cos(phi1 - phi2) - 11*(y1 - y2)*(cos(phi1) + cos(phi2)) + 11*(x1 - x2)*(sin(phi1) + sin(phi2)))/(4.*l0*one_p_cos_dphi);
	double aR   = (64*a1 + 64*a2 - 5*l0*phi1 + 5*l0*phi2 + (64*a1 + 64*a2 + 5*l0*(-phi1 + phi2))*cos(phi1 - phi2) + 36*(y1 - y2)*(cos(phi1) + cos(phi2)) - 36*(x1 - x2)*(sin(phi1) + sin(phi2)))/(72.*one_p_cos_dphi);
	double bR   = (2*(8*a1 - 8*a2 + l0*(-phi1 + phi2)) + 2*(8*a1 - 8*a2 + l0*(-phi1 + phi2))*cos(phi1 - phi2) + 11*(y1 - y2)*(cos(phi1) + cos(phi2)) - 11*(x1 - x2)*(sin(phi1) + sin(phi2)))/(4.*l0*one_p_cos_dphi);

// Lage-Koordinaten berechnen
    if (s < 0.0) // linker Teilbereich
    {
 	XLocal[0] = xS + (1 + eps)*s*cos(bL - phiS) + (s*(bL*l0*(2*l0 - 5*s)*Power(l0 + 2*s,2) + s*(-((8*aR - 3*bR*l0)*Power(l0 + 2*s,2)) - 8*aL*(l0h2 - 4*l0*s - 8*Power(s,2))))*sin(bL - phiS))/(2.*l0h4);
 	XLocal[1] = yS + (s*(bL*l0*(2*l0 - 5*s)*Power(l0 + 2*s,2) + s*(-((8*aR - 3*bR*l0)*Power(l0 + 2*s,2)) - 8*aL*(l0h2 - 4*l0*s - 8*Power(s,2))))*cos(bL - phiS))/(2.*l0h4) - (1 + eps)*s*sin(bL - phiS);
//	XLocal[2] = phiS + (s*(l0h2*(-8*(aL + aR) + 3*(bL + bR)*l0) - 6*l0*(-8*aL + 8*aR + 3*(bL - bR)*l0)*s + 8*(16*aL - 8*aR - 5*bL*l0 + 3*bR*l0)*Power(s,2)))/l0h4;
    }
    else       // rechter Teilbereich
    {
 	XLocal[0] = xS + (1 + eps)*s*cos(bR + phiS) + (s*(bR*l0*Power(l0 - 2*s,2)*(2*l0 + 5*s) + s*(8*aL*Power(l0 - 2*s,2) - 3*bL*l0*Power(l0 - 2*s,2) + 8*aR*(l0h2 + 4*l0*s - 8*Power(s,2))))*sin(bR + phiS))/(2.*l0h4);
 	XLocal[1] = yS - (s*(bR*l0*Power(l0 - 2*s,2)*(2*l0 + 5*s) + s*(8*aL*Power(l0 - 2*s,2) - 3*bL*l0*Power(l0 - 2*s,2) + 8*aR*(l0h2 + 4*l0*s - 8*Power(s,2))))*cos(bR + phiS))/(2.*l0h4) + (1 + eps)*s*sin(bR + phiS);
//	XLocal[2] = phiS + (s*(l0h2*(-8*(aL + aR) + 3*(bL + bR)*l0) - 6*l0*(-8*aL + 8*aR + 3*(bL - bR)*l0)*s - 8*(8*aL - 16*aR - 3*bL*l0 + 5*bR*l0)*Power(s,2)))/l0h4;
    }

    X[0] = JT[0][0]*XLocal[0] + JT[0][1]*XLocal[1] + WrN00[0];
    X[1] = JT[1][0]*XLocal[0] + JT[1][1]*XLocal[1] + WrN00[1];
    X[2] = JT[2][0]*XLocal[0] + JT[2][1]*XLocal[1] + WrN00[2];
}

void ElasticBody1s21RCM::initCoordinates() {
    for(int i=0;i<nElements;i++) {
        q[5*i+0] = i;
        for(int j=1;j<5;j++) q[5*i+j] = 0.0;	    
    }
    if(!closed) {
	q[5*nElements+0] = nElements;
	q[5*nElements+1] = 0.0;
 	q[5*nElements+2] = 0.0;
    } 
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
double ElasticBody1s21RCM::Power(double base, int exponent)
{
    return pow(base,exponent);
}

