/*
 * AMVis - 3D Multibody Visualisation Program.
 * Copyright (C) 2006 Institute of Applied Mechanics,
 *                    Technische Universitaet Muenchen

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* 
 * (c) 2006-08  Thorsten Schindler
 * Contact: schindler@amm.mw.tum.de
 * 
 * VERSION 22.01.08
 */

#include "elastic1s33rcm.h"
using namespace std;

ElasticBody1s33RCM::ElasticBody1s33RCM(char *body_name_,char *pos_file_) : ElasticBody1s(body_name_,pos_file_)
{
	/*
	 *  body-file structure
	 *  ElasticBody1s33RCM	CLASS NAME
	 *	1					INSTANCES
	 *	1					ASCII (0) OR BINARY (1) POS-FILE
	 *	1					NUMBER OF FE-BEAMS
	 *	o					OPEN (o) OR CLOSED (c) STRUCTURE
	 *	l0					BEAM LENGTH
	 *	0 0 0				INITIAL TRANSLATION
	 *	1 0 0
	 *	0 1 0
	 *	0 0 1				JACOBIAN OF TRAFO INTO INERTIAL FOR FROM BODY INERTIAL FOR
	 *	1 -1				CYLINDER RADIUS or CUBOID BREADTH and CUBOID HEIGHT
	 */
	
    int result;
    char *charstringline = new char[charlength];
  
  	for(int i=0;i<16;i++) qElement[i] = 0.;
  
    /* reading body-file begins with NUMBER OF FE-BEAMS */    
    /* number of FE-Beams */
    parsenextdataline(datafile);
    result = GetCharacterLine(charstringline,charlength,datafile);
    sscanf(charstringline, "%i" ,&nElements);
      
    /* open or closed structure */
    char cclosed;
    parsenextdataline(datafile);
    result = GetCharacterLine(charstringline,charlength,datafile);
    sscanf(charstringline, "%c" ,&cclosed);
    if (cclosed == 'c') closed = true;
    else closed = false;
    
    /* setting dof for AMVis */
    if(closed) {
     	dofs = 10*nElements;
        binaryrecordsize=sizeof(float)*(dofs + 1); // dofs with time t
        q = new float[dofs];
    }
    else {
       	dofs = 10*nElements + 6;
        binaryrecordsize=sizeof(float)*(dofs + 1); // dofs with time t
        q = new float[dofs];
    }
      
    /* length of entire beam */
    parsenextdataline(datafile);
    result = GetCharacterLine(charstringline,charlength,datafile);
    sscanf(charstringline, "%f" ,&l0);
	lGes    = nElements*l0; // length of entire beam
    lOffSet = lGes * 0.5e-5; // length offsets for correct drawing
    lGesEff = lGes - 2*lOffSet;     
    l0h2 = l0*l0; l0h3 = l0h2*l0; l0h4 = l0h2*l0h2; l0h5 = l0h3*l0h2;
    
    /* WrN00 */
  	parsenextdataline(datafile);
  	result = GetCharacterLine(charstringline,charlength,datafile);
  	sscanf(charstringline, "%f %f %f" ,&WrN00[0],&WrN00[1],&WrN00[2]);

  	/* Jac */
  	parsenextdataline(datafile); // first row
  	result = GetCharacterLine(charstringline,charlength,datafile);
  	sscanf(charstringline, "%f %f %f" ,&Jac[0][0],&Jac[0][1],&Jac[0][2]);
  	parsenextdataline(datafile); // second row
  	result = GetCharacterLine(charstringline,charlength,datafile);
  	sscanf(charstringline, "%f %f %f" ,&Jac[1][0],&Jac[1][1],&Jac[1][2]);
  	parsenextdataline(datafile); // third row
  	result = GetCharacterLine(charstringline,charlength,datafile);
  	sscanf(charstringline, "%f %f %f" ,&Jac[2][0],&Jac[2][1],&Jac[2][2]);
  	
  	/* cylinder (see http://pyopengl.sourceforge.net/documentation/manual/index.xml) */
  	float cylinderRadius, cuboidHeight;
  	parsenextdataline(datafile);
	result = GetCharacterLine(charstringline,charlength,datafile);
	sscanf(charstringline, "%f %f" ,&cylinderRadius,&cuboidHeight);
	if (cylinderRadius > 0. && cuboidHeight == -1.) {
		nContourPoints = 33;
  
  		cont_normal = new SVec[nContourPoints];
  		contour     = new SVec[nContourPoints];
  
  		for(int i=0; i<nContourPoints; i++) {
      		cont_normal[i][0] = cos(2*M_PI/(nContourPoints-1)*i);
      		cont_normal[i][1] = sin(2*M_PI/(nContourPoints-1)*i);
      		contour[i][0]     = cylinderRadius*cont_normal[i][0];
      		contour[i][1]     = cylinderRadius*cont_normal[i][1];
  		}
  		up[1] = 1.;
	}
	else if (cylinderRadius>0. && cuboidHeight>0.) {
		breadth = cylinderRadius;
		height = cuboidHeight;
		      
    	nContourPoints = 8; // 4 corners of rectangular belong to two sides, respectively
    	cont_normal = new SVec[nContourPoints]; // (y,z)
    	contour     = new SVec[nContourPoints]; // (y,z)
        
    	contour[0][0] =  0.5*breadth;  cont_normal[0][0]     =  0.;  
    	contour[0][1] =  0.5*height;  cont_normal[0][1]     =  1.;
      
    	contour[1][0] = -0.5*breadth;  cont_normal[1][0]     =  0.; 
    	contour[1][1] =  0.5*height;  cont_normal[1][1]     =  1.;
      
    	contour[2][0] = -0.5*breadth;  cont_normal[2][0]     = -1.; 
    	contour[2][1] =  0.5*height;  cont_normal[2][1]     =  0.;
      
    	contour[3][0] = -0.5*breadth;  cont_normal[3][0]     = -1.; 
    	contour[3][1] = -0.5*height;  cont_normal[3][1]     =  0.;
      
    	contour[4][0] = -0.5*breadth;  cont_normal[4][0]     =  0.; 
    	contour[4][1] = -0.5*height;  cont_normal[4][1]     = -1.;
      
    	contour[5][0] =  0.5*breadth;  cont_normal[5][0]     =  0.; 
    	contour[5][1] = -0.5*height;  cont_normal[5][1]     = -1.;
      
    	contour[6][0] =  0.5*breadth;  cont_normal[6][0]     =  1.; 
    	contour[6][1] = -0.5*height;  cont_normal[6][1]     =  0.;
      
    	contour[7][0] =  0.5*breadth;  cont_normal[7][0]     =  1.; 
    	contour[7][1] =  0.5*height;  cont_normal[7][1]     =  0.;
      
    	up[0] = Jac[0][1]; up[1] = Jac[1][1]; up[2] = Jac[2][1]; // y-axis is up direction in inertial FR
	}
	else {
		nContourPoints = 33;
  
  		cont_normal = new SVec[nContourPoints];
  		contour     = new SVec[nContourPoints];
  
  		for(int i=0; i<nContourPoints; i++) {
      		cont_normal[i][0] = cos(2*M_PI/(nContourPoints-1)*i);
      		cont_normal[i][1] = sin(2*M_PI/(nContourPoints-1)*i);
      		contour[i][0]     = 0.001*cont_normal[i][0];
      		contour[i][1]     = 0.001*cont_normal[i][1];
  		}
  		up[1] = 1.0;
	}
}

ElasticBody1s33RCM::~ElasticBody1s33RCM() {}

void ElasticBody1s33RCM::computeSweep(const int npoints,gleDouble point_array[][3],gleDouble twist_array[])
{
    double ds = lGesEff/(npoints-1); // sampling distance      
    GLdouble XTemp[6];
        
    /* startPoint, not drawn */
    LocateStructure(XTemp,0.); 
    point_array[0][0] = Jac[0][0]*XTemp[0]+Jac[0][1]*XTemp[1]+Jac[0][2]*XTemp[2]+WrN00[0]; // Global position
    point_array[0][1] = Jac[1][0]*XTemp[0]+Jac[1][1]*XTemp[1]+Jac[1][2]*XTemp[2]+WrN00[1];
    point_array[0][2] = Jac[2][0]*XTemp[0]+Jac[2][1]*XTemp[1]+Jac[2][2]*XTemp[2]+WrN00[2];
    
    twist_array[0]    = 180./M_PI * XTemp[3]; // local twist around axis: GLE needs degree for input
  
    for(int i=0;i<npoints;i++) {
       	LocateStructure(XTemp,lOffSet + ds*i); 
        point_array[i+1][0] = Jac[0][0]*XTemp[0]+Jac[0][1]*XTemp[1]+Jac[0][2]*XTemp[2]+WrN00[0];
        point_array[i+1][1] = Jac[1][0]*XTemp[0]+Jac[1][1]*XTemp[1]+Jac[1][2]*XTemp[2]+WrN00[1];
        point_array[i+1][2] = Jac[2][0]*XTemp[0]+Jac[2][1]*XTemp[1]+Jac[2][2]*XTemp[2]+WrN00[2];
 
    	twist_array[i+1]    = 180./M_PI * XTemp[3];
    }

    /* endPoint, not drawn */
    LocateStructure(XTemp,lGes); 
    point_array[npoints+1][0] = Jac[0][0]*XTemp[0]+Jac[0][1]*XTemp[1]+Jac[0][2]*XTemp[2]+WrN00[0];
    point_array[npoints+1][1] = Jac[1][0]*XTemp[0]+Jac[1][1]*XTemp[1]+Jac[1][2]*XTemp[2]+WrN00[1];
    point_array[npoints+1][2] = Jac[2][0]*XTemp[0]+Jac[2][1]*XTemp[1]+Jac[2][2]*XTemp[2]+WrN00[2];

    twist_array[npoints+1]    = 180./M_PI * XTemp[3];
    
    /* consistency for twist_array */
    for(int i=0;i<npoints+1;i++) {
    	while(fabs(twist_array[i+1]-twist_array[i])>300.) {
    		if(twist_array[i]<0.) twist_array[i+1] = twist_array[i+1] - 360.;
    		if(twist_array[i]>0.) twist_array[i+1] = twist_array[i+1] + 360.;
    	}
    }
}

void ElasticBody1s33RCM::ret_rot(float *rotation)
{
    rotation[0] = 0.; // no rotation of the camera
    rotation[1] = 0.;
    rotation[2] = 0.;
}

void ElasticBody1s33RCM::ret_trans(float *translation)
{
	translation[0] = Jac[0][0]*q[0] + Jac[0][1]*q[1] + Jac[0][2]*q[2] + WrN00[0]; // position of the beam beginning
    translation[1] = Jac[1][0]*q[0] + Jac[1][1]*q[1] + Jac[1][2]*q[2] + WrN00[1];
    translation[2] = Jac[2][0]*q[0] + Jac[2][1]*q[1] + Jac[2][2]*q[2] + WrN00[2];
}

void ElasticBody1s33RCM::init()
{
	initCoordinates();
   	ElasticBody1s::init();
}

void ElasticBody1s33RCM::return_info_string(QString& info)
{
	if(closed) {
        info=tr("Class: %1\nBody-File: %2\nPos-File: %3\nRing\nElements: %4\nl_0: %5").arg(class_name).arg(body_file_name).arg(posfile_name).arg(nElements).arg(l0);
    }
    else {
        info=tr("Class: %1\nBody-File: %2\nPos-File: %3\nBeam\nElements: %4\nl_0: %5").arg(class_name).arg(body_file_name).arg(posfile_name).arg(nElements).arg(l0);
    }
}

void ElasticBody1s33RCM::LocateStructure(GLdouble X[6],const double s)
{
	/* selecting the FE-Beam affected by s */
	int nElement = 0; 
	while((nElement+1)*l0 < s) nElement++;	
	double sElement = s - nElement * l0 - l0/2.; // Lagrange-Parameter of the affected FE with sLocal==0 in the middle of the FE and s==0 at the beginning of the beam
    
	if (nElements <= nElement && closed) nElement -= nElements; // correction for numerical errors
    else if (nElements <= nElement) {
		nElement  -= 1;
		sElement +=l0;
    }
	
	/* setting the global coordinates of the FE */
	for(int i=0;i<16;i++) {
		if(i<10 || !closed || nElement != nElements-1) qElement[i] = q[10*nElement+i]; 
		else if(i!=15) qElement[i] = q[i-10];
		else if (q[10*nElement+5]<q[5]) qElement[15] = q[5]-2*M_PI; // ring only around z-axis
		else qElement[15] = q[5]+2*M_PI;
	}
	
	compute_State(X,qElement,sElement);
}

void ElasticBody1s33RCM::initCoordinates()
{
    for(int i=0;i<nElements;i++) {
        q[10*i+0] = i;
        for(int j=1;j<10;j++) q[10*i+j] = 0.;	    
    }
    if(!closed) {
        q[10*nElements+0] = nElements; 
        q[10*nElements+1] = 0.;
        q[10*nElements+2] = 0.;
		q[10*nElements+3] = 0.;
        q[10*nElements+4] = 0.;
	    q[10*nElements+5] = 0.;
    } 
}

void ElasticBody1s33RCM::compute_State(GLdouble X[6],double qG[16],double x)
{	
	AMVisRevKardan* angle = new AMVisRevKardan();
	AMVisLA* la = new AMVisLA();
		
	/* Transformation */
	double rRrLm[3];
	rRrLm[0] = qG[10]-qG[0];
	rRrLm[1] = qG[11]-qG[1];
	rRrLm[2] = qG[12]-qG[2];
		
	double pM[3];
	pM[0] = 0.5*(qG[3]+qG[13]);
	pM[1] = 0.5*(qG[4]+qG[14]);
	pM[2] = 0.5*(qG[5]+qG[15]);
	
	double tM[3];
	angle->compute_t(tM,pM);
	
	double nM[3];
	angle->compute_n(nM,pM);
	
	double bM[3];
	angle->compute_b(bM,pM);
	
	double nMtil[3];
	angle->compute_ntil(nMtil,pM);
	
	double bMtil[3];
	angle->compute_btil(bMtil,pM);
			
	double nMpM[3][3];
	angle->compute_np(nMpM,pM);
	
	double bMpM[3][3];
	angle->compute_bp(bMpM,pM);
	
	double xinMtil = la->dot_Product(nM,nMtil,3);
	double xibMtil = la->dot_Product(nM,bMtil,3);
	double etanMtil = la->dot_Product(bM,nMtil,3);
	double etabMtil = la->dot_Product(bM,bMtil,3);
	
	double SMRHS[8][9];
 	for(int i=0;i<8;i++) {
 		for(int j=0;j<9;j++) SMRHS[i][j] = 0.;
 	}
			
	SMRHS[0][2] = -1.; SMRHS[0][3] = 1.;
	SMRHS[1][6] = -1.; SMRHS[1][7] = 1.;
	SMRHS[2][0] = -xinMtil; SMRHS[2][1] = xinMtil; SMRHS[2][4] = -xibMtil; SMRHS[2][5] = xibMtil;
	
	double nMpMVec[3];
	nMpMVec[0] = nMpM[0][0];
	nMpMVec[1] = nMpM[1][0];
	nMpMVec[2] = nMpM[2][0];	
	double rRrLmnMpM;
	rRrLmnMpM = 0.5*la->dot_Product(rRrLm,nMpMVec,3);
	SMRHS[2][6] = sin(pM[1])*rRrLmnMpM; SMRHS[2][7] = sin(pM[1])*rRrLmnMpM;
	
	nMpMVec[0] = nMpM[0][1];
	nMpMVec[1] = nMpM[1][1];
	nMpMVec[2] = nMpM[2][1];
	rRrLmnMpM = 0.5*la->dot_Product(rRrLm,nMpMVec,3);
	SMRHS[2][2] = rRrLmnMpM; SMRHS[2][3] = rRrLmnMpM;
	
	nMpMVec[0] = nMpM[0][2];
	nMpMVec[1] = nMpM[1][2];
	nMpMVec[2] = nMpM[2][2];
	rRrLmnMpM = 0.5*la->dot_Product(rRrLm,nMpMVec,3);
	SMRHS[2][6] += rRrLmnMpM; SMRHS[2][7] += rRrLmnMpM;
		
	SMRHS[3][0] = -etanMtil; SMRHS[3][1] = etanMtil; SMRHS[3][4] = -etabMtil; SMRHS[3][5] = etabMtil;
	
	double bMpMVec[3];
	bMpMVec[0] = bMpM[0][0];
	bMpMVec[1] = bMpM[1][0];
	bMpMVec[2] = bMpM[2][0];	
	double rRrLmbMpM;
	rRrLmbMpM = 0.5*la->dot_Product(rRrLm,bMpMVec,3);
	SMRHS[3][6] = sin(pM[1])*rRrLmbMpM; SMRHS[3][7] = sin(pM[1])*rRrLmbMpM;
	
	bMpMVec[0] = bMpM[0][1];
	bMpMVec[1] = bMpM[1][1];
	bMpMVec[2] = bMpM[2][1];
	rRrLmbMpM = 0.5*la->dot_Product(rRrLm,bMpMVec,3);
	SMRHS[3][2] = rRrLmbMpM; SMRHS[3][3] = rRrLmbMpM;
	
	bMpMVec[0] = bMpM[0][2];
	bMpMVec[1] = bMpM[1][2];
	bMpMVec[2] = bMpM[2][2];
	rRrLmbMpM = 0.5*la->dot_Product(rRrLm,bMpMVec,3);
	SMRHS[3][6] += rRrLmbMpM; SMRHS[3][7] += rRrLmbMpM;
	
	SMRHS[4][0] = 7./16.; SMRHS[4][1] = 7./16.; SMRHS[4][2] = 3.*l0/64.; SMRHS[4][3] = -3.*l0/64.;
	SMRHS[5][0] = -17./64.; SMRHS[5][1] = 17./64.; SMRHS[5][2] = -3.*l0/128.; SMRHS[5][3] = -3.*l0/128.;
	SMRHS[6][4] = 7./16.; SMRHS[6][5] = 7./16.; SMRHS[6][6] = 3.*l0/64.; SMRHS[6][7] = -3.*l0/64.;
	SMRHS[7][4] = -17./64.; SMRHS[7][5] = 17./64.; SMRHS[7][6] = -3.*l0/128.; SMRHS[7][7] = -3.*l0/128.;
		
	SMRHS[0][8] = qG[14]-qG[4];
	SMRHS[1][8] = qG[15]-qG[5];
	SMRHS[2][8] = la->dot_Product(nM,rRrLm,3);	
	SMRHS[3][8] = la->dot_Product(bM,rRrLm,3);
	SMRHS[4][8] = qG[6]+qG[7];
	SMRHS[5][8] = qG[7]-qG[6];
	SMRHS[6][8] = qG[8]+qG[9];
	SMRHS[7][8] = qG[9]-qG[8];
 	
 	double be[8];
 	la->Gaussalg(&SMRHS[0][0],8,&be[0]);
	
	double qI[16];
	qI[14] = be[7];
	qI[13] = be[6];
	qI[12] = be[5]; 
	qI[11] = be[4]; 
	qI[10] = be[3]; 
	qI[9] = be[2];
	qI[8] = be[1]; 
	qI[7] = be[0];
	
	double pS[3];
	pS[1] = 0.5*((qG[4]+qG[14])-(be[2]+be[3]));
	pS[2] = 0.5*((qG[5]+qG[15])-(be[6]+be[7]));
	pS[0] = 0.5*(qG[3]+qG[13]-sin(pS[1])*(be[6]+be[7]));
	
	double tS[3];
	angle->compute_t(tS,pS);
	
	double nS[3];
	angle->compute_n(nS,pS);
	
	double bS[3];
	angle->compute_b(bS,pS);
	
	double nStil[3];
	angle->compute_ntil(nStil,pS);
	
	double bStil[3];
	angle->compute_btil(bStil,pS);
	
	double xinStil = la->dot_Product(nS,nStil,3);
	double xibStil = la->dot_Product(nS,bStil,3);
	double etanStil = la->dot_Product(bS,nStil,3);
	double etabStil = la->dot_Product(bS,bStil,3);
	
	qI[15] = 1/l0*(qG[13]-qG[3]-sin(pS[1])*(be[7]-be[6]));
	qI[6] = la->dot_Product(tS,rRrLm,3)/l0-1.;
	
	qI[5] = pS[2];
	qI[4] = pS[1];
	qI[3] = pS[0];
	
	qI[2] = 0.5*((qG[2]+qG[12])-(xinStil*(be[0]+be[1])+xibStil*(be[4]+be[5]))*nS[2]-(etanStil*(be[0]+be[1])+etabStil*(be[4]+be[5]))*bS[2]);
	qI[1] = 0.5*((qG[1]+qG[11])-(xinStil*(be[0]+be[1])+xibStil*(be[4]+be[5]))*nS[1]-(etanStil*(be[0]+be[1])+etabStil*(be[4]+be[5]))*bS[1]);
	qI[0] = 0.5*((qG[0]+qG[10])-(xinStil*(be[0]+be[1])+xibStil*(be[4]+be[5]))*nS[0]-(etanStil*(be[0]+be[1])+etabStil*(be[4]+be[5]))*bS[0]);
	
	/* state calculation */
	double w1coef[4];
	compute_wcoef(w1coef,qI[7],qI[8],qI[9],qI[10]);
	
	double w2coef[4];
	compute_wcoef(w2coef,qI[11],qI[12],qI[13],qI[14]);
	
	double w1[2];
	compute_w(w1,w1coef,x);
	
	double w2[2];
	compute_w(w2,w2coef,x);
	
	double wh1 = xinStil*w1[0]+xibStil*w2[0];
	double wh2 = etanStil*w1[0]+etabStil*w2[0];
	
	X[0] = qI[0]+(1.+qI[6])*x*tS[0]+wh1*nS[0]+wh2*bS[0];
	X[1] = qI[1]+(1.+qI[6])*x*tS[1]+wh1*nS[1]+wh2*bS[1];
	X[2] = qI[2]+(1.+qI[6])*x*tS[2]+wh1*nS[2]+wh2*bS[2];	
	X[3] = qI[3]+sin(pS[1])*w2[1]+qI[15]*x;
	X[4] = qI[4]+w1[1];
	X[5] = qI[5]+w2[1];
	
	delete angle;
	delete la;
}

void ElasticBody1s33RCM::compute_w(double W[2],double wt[4],double x)
{
	W[0] = (((wt[0]*x+wt[1])*x+wt[2])*x+wt[3])*x*x;
	W[1] = (((5*wt[0]*x+4*wt[1])*x+3*wt[2])*x+2*wt[3])*x;
}

void ElasticBody1s33RCM::compute_wcoef(double wt[4],double dL,double dR,double bL,double bR)
{
	double dLdRp = dL+dR;
  	double dLdRm = dL-dR;
  	double bLbRp = bL+bR;
  	double bLbRm = bL-bR;
  	
  	wt[0] = (4.*l0*bLbRp+24.*dLdRm)/l0h5;
  	wt[1] = (-2.*l0*bLbRm-8.*dLdRp)/l0h4;
  	wt[2] = (-l0*bLbRp-10.*dLdRm)/l0h3;
  	wt[3] = (l0*bLbRm+8.*dLdRp)/(2.*l0h2);
  		 	
}
/*******************************************************************/

/* CLASS AMVisLA */
AMVisLA::AMVisLA() {}
AMVisLA::~AMVisLA() {}

double AMVisLA::dot_Product(double *v1,double *v2,int size) const
{
	double dp = 0.;
	for(int i=0;i<size;i++)	dp += v1[i]*v2[i];
	return dp;
}

void AMVisLA::Gaussalg(double *Ab,int n,double *x) const
{
  	/* elimination steps */
  	for(int s=0;s<n;s++) {
    	double maximum = fabs(Ab[s*(n+1)+s]); // column pivoting with rowwise matrix architecture
    	int pzeile = s; // pivot row 
    	for(int i=s+1;i<n;i++) {
      		if(fabs(Ab[i*(n+1)+s]) > maximum) {
        		maximum = fabs(Ab[i*(n+1)+s]);
        		pzeile = i;
      		}
    	}
    	if(pzeile != s) {
      		for(int j=s;j<(n+1);j++) {
        		double h = Ab[s*(n+1)+j];
        		Ab[s*(n+1)+j] = Ab[pzeile*(n+1)+j];
        		Ab[pzeile*(n+1)+j]= h;
      		}
    	}
    	for(int i=s+1;i<n;i++) { // elimination 
      		double f = -(Ab[i*(n+1)+s]/Ab[s*(n+1)+s]); // factor of multiplication
      		Ab[i*(n+1)+s] = 0.0;
      		for(int j=s+1;j<(n+1);j++) Ab[i*(n+1)+j] += f*Ab[s*(n+1)+j];
    	}
  	}

  	/* backsubstitution */  
    x[n-1] =  Ab[(n-1)*(n+1)+n]/Ab[(n-1)*(n+1)+(n-1)];
    for(int i=n-2;i>=0;i--) {
      	for(int j=n-1;j>i;j--) {
       		Ab[i*(n+1)+n] -= x[j]*Ab[i*(n+1)+j];
     	} 
     	x[i] = Ab[i*(n+1)+n]/Ab[i*(n+1)+i];
    } 
}
/*******************************************************************/

/* CLASS AMVisKardan */
AMVisKardan::AMVisKardan() {}
AMVisKardan::~AMVisKardan() {}

void AMVisKardan::compute_t(double t[3],double p[3]) const
{
	t[0] = cos(p[1])*cos(p[2]);
	t[1] = cos(p[0])*sin(p[2])+sin(p[0])*sin(p[1])*cos(p[2]);
 	t[2] = sin(p[0])*sin(p[2])-cos(p[0])*sin(p[1])*cos(p[2]);
}
 
void AMVisKardan::compute_n(double n[3],double p[3]) const
{
 	n[0] = -cos(p[1])*sin(p[2]);
 	n[1] =  cos(p[0])*cos(p[2])-sin(p[0])*sin(p[1])*sin(p[2]);
 	n[2] =  sin(p[0])*cos(p[2])+cos(p[0])*sin(p[1])*sin(p[2]);
}
 
void AMVisKardan::compute_b(double b[3],double p[3]) const
{
 	b[0] =  sin(p[1]);
 	b[1] = -sin(p[0])*cos(p[1]);
 	b[2] =  cos(p[0])*cos(p[1]);
}

void AMVisKardan::compute_ntil(double ntil[3],double p[3]) const
{
 	ntil[0] = -sin(p[1])*cos(p[2]);
 	ntil[1] =  sin(p[0])*cos(p[1])*cos(p[2]);
 	ntil[2] =  -cos(p[0])*cos(p[1])*cos(p[2]);
}

void AMVisKardan::compute_btil(double btil[3],double p[3]) const
{
 	btil[0] = -cos(p[1])*sin(p[2]);
 	btil[1] =  cos(p[0])*cos(p[2])-sin(p[0])*sin(p[1])*sin(p[2]);
 	btil[2] =  sin(p[0])*cos(p[2])+cos(p[0])*sin(p[1])*sin(p[2]);
}

void AMVisKardan::compute_np(double np[3][3],double p[3]) const
{
 	np[0][0] = 0.;
 	np[0][1] = sin(p[1])*sin(p[2]);
 	np[0][2] = -cos(p[1])*cos(p[2]);
 	np[1][0] = -sin(p[0])*cos(p[2])-cos(p[0])*sin(p[1])*sin(p[2]); 
 	np[1][1] = -sin(p[0])*cos(p[1])*sin(p[2]); 
 	np[1][2] = -cos(p[0])*sin(p[2])-sin(p[0])*sin(p[1])*cos(p[2]);
 	np[2][0] = cos(p[0])*cos(p[2])-sin(p[0])*sin(p[1])*sin(p[2]); 
 	np[2][1] = cos(p[0])*cos(p[1])*sin(p[2]); 
 	np[2][2] = -sin(p[0])*sin(p[2])+cos(p[0])*sin(p[1])*cos(p[2]);	
}
  
void AMVisKardan::compute_bp(double bp[3][3],double p[3]) const
{
 	bp[0][0] = 0.;
 	bp[0][1] = cos(p[1]);
 	bp[0][2] = 0.;
 	bp[1][0] = -cos(p[0])*cos(p[1]);
 	bp[1][1] = sin(p[0])*sin(p[1]);
 	bp[1][2] = 0.;
 	bp[2][0] = -sin(p[0])*cos(p[1]);
 	bp[2][1] = -cos(p[0])*sin(p[1]);
 	bp[2][2] = 0.;	
}
/*******************************************************************/

/* CLASS AMVisRevKardan */
AMVisRevKardan::AMVisRevKardan() {}
AMVisRevKardan::~AMVisRevKardan() {}

void AMVisRevKardan::compute_t(double t[3],double p[3]) const
{
	t[0] = cos(p[2])*cos(p[1]);
	t[1] = sin(p[2])*cos(p[1]);
 	t[2] = -sin(p[1]);
}
 
void AMVisRevKardan::compute_n(double n[3],double p[3]) const
{
 	n[0] = -sin(p[2])*cos(p[0])+cos(p[2])*sin(p[1])*sin(p[0]);
 	n[1] =  cos(p[0])*cos(p[2])+sin(p[0])*sin(p[1])*sin(p[2]);
 	n[2] =  sin(p[0])*cos(p[1]);
}
 
void AMVisRevKardan::compute_b(double b[3],double p[3]) const
{
 	b[0] =  sin(p[2])*sin(p[0])+cos(p[2])*sin(p[1])*cos(p[0]);
 	b[1] = -sin(p[0])*cos(p[2])+sin(p[2])*sin(p[1])*cos(p[0]);
 	b[2] =  cos(p[0])*cos(p[1]);
}

void AMVisRevKardan::compute_ntil(double ntil[3],double p[3]) const
{
 	ntil[0] = -sin(p[1])*cos(p[2]);
 	ntil[1] =  -sin(p[2])*sin(p[1]);
 	ntil[2] =  -cos(p[1]);
}

void AMVisRevKardan::compute_btil(double btil[3],double p[3]) const
{
 	btil[0] = -cos(p[1])*sin(p[2]);
 	btil[1] =  cos(p[1])*cos(p[2]);
 	btil[2] =  0.;
}

void AMVisRevKardan::compute_np(double np[3][3],double p[3]) const
{
 	np[0][0] = sin(p[2])*sin(p[0])+cos(p[2])*sin(p[1])*cos(p[0]);
 	np[0][1] = cos(p[2])*cos(p[1])*sin(p[0]);
 	np[0][2] = -cos(p[0])*cos(p[2])-sin(p[2])*sin(p[1])*sin(p[0]);
 	np[1][0] = -sin(p[0])*cos(p[2])+cos(p[0])*sin(p[1])*sin(p[2]); 
 	np[1][1] = sin(p[0])*cos(p[1])*sin(p[2]); 
 	np[1][2] = -cos(p[0])*sin(p[2])+sin(p[0])*sin(p[1])*cos(p[2]);
 	np[2][0] = cos(p[0])*cos(p[1]); 
 	np[2][1] = -sin(p[0])*sin(p[1]); 
 	np[2][2] = 0.;	
}
  
void AMVisRevKardan::compute_bp(double bp[3][3],double p[3]) const
{
 	bp[0][0] = sin(p[2])*cos(p[0])-cos(p[2])*sin(p[1])*sin(p[0]); 
 	bp[0][1] = cos(p[2])*cos(p[1])*cos(p[0]); 
 	bp[0][2] = cos(p[2])*sin(p[0])-sin(p[2])*sin(p[1])*cos(p[0]);
 	bp[1][0] = -cos(p[0])*cos(p[2])-sin(p[2])*sin(p[1])*sin(p[0]); 
 	bp[1][1] = sin(p[2])*cos(p[1])*cos(p[0]); 
 	bp[1][2] = sin(p[2])*sin(p[0])+cos(p[2])*sin(p[1])*cos(p[0]);
 	bp[2][0] = -sin(p[0])*cos(p[1]); 
 	bp[2][1] = -cos(p[0])*sin(p[1]); 
 	bp[2][2] = 0.;	
}
/*******************************************************************/
