/*
 * AMVis - 3D Multibody Visualisation Program.
 * Copyright (C) 2006 Institute of Applied Mechanics,
 *                    Technische Universitaet Muenchen

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* 
 * (c) 2006-08  Thorsten Schindler
 * Contact: schindler@amm.mw.tum.de
 * 
 * VERSION 14.01.08
 */

#ifndef _ELASTIC_BODY_1S_33_RCM_H_
#define _ELASTIC_BODY_1S_33_RCM_H_

#include "elastic.h"

/*! \brief Visualisation for 3D-Beam using Redundant Coordinate Method (RCM) and OpenGL */
class ElasticBody1s33RCM : public ElasticBody1s {
	public:
		/*! Constructor */
    	ElasticBody1s33RCM(char *body_name_, char *pos_file_);
    	/*! Destructor */
    	virtual ~ElasticBody1s33RCM();

		/*! Computes the position of the entire beam cross-sections	*/
    	void computeSweep(const int npoints,gleDouble point_array[][3],gleDouble twist_array[]);  	
    	/*! Defines the rotational position of the camera moving with body */
    	void ret_rot(float *rotation);   	
    	/*! Defines the translational position of the camera moving with body */
    	void ret_trans(float *translation);    	
    	/*! Initialises AMVis */
    	void init();   	
    	/*! Sets the text in the left info box of AMVis */
    	void return_info_string(QString& info);

	protected:
		/** number of FE */
        int nElements;
        
        /** closed or open entire structure */
        bool closed;
        
        /** breadth and height of the FE for drawing with AMVis */
        float breadth, height;
        
        /** length of FE-Beam */
        float l0, l0h2, l0h3, l0h4, l0h5;
        
        /** length offsets of entire beam necessary for drawing */
        double lGes,lGesEff,lOffSet;
        
        /** inertial body FoR with respect to inertial FoR */
		float WrN00[3];
   		float Jac[3][3];

		/*! Returns the State at the Lagrange-Parameter s of the entire beam */
        void LocateStructure(GLdouble X[6],const double s);
        /*! Numbers the elements in the coordinates of the entire beam */
        void initCoordinates();

 	private:
 		/** global coordinates of one FE-Beam */
        double qElement[16];
        
        /*! Computes the beam state in global coordinates */
		void compute_State(GLdouble X[6],double qG[16],double x);
		/*! Evaluates the bending polynomial and its x-derivative at \param x */
		void compute_w(double W[2],double wt[4],double x);
		/*! Computes the coefficients of w */
		void compute_wcoef(double wt[4],double dL,double dR,double bL,double bR);
};
/*******************************************************************/

/*! Linear Algebra Package for AMVis */
class AMVisLA {
	public:
		/*! Constructor */
		AMVisLA();
		/*! Destructor */
		virtual ~AMVisLA();
		
		/*! Dotproduct of the two vectors \param v1 and \param v2 of dimension \param size */
		double dot_Product(double* v1,double* v2,int size) const;
		/*! Solves A*x=b of dimension \param n with Gaussian algorithm */
		void Gaussalg(double* Ab,int n,double* x) const;		
};
/*******************************************************************/

/*! Angle parametrisation for AMVis */
class AMVisAngle {
	public:
		/*! Constructor */
		AMVisAngle();
		/*! Destructor */
		virtual ~AMVisAngle();
		
		/*! Computes tangent */
		virtual void compute_t(double t[3],double p[3]) const = 0;
		/*! Computes normal */
		virtual void compute_n(double n[3],double p[3]) const = 0;
		/*! Computes binormal */
		virtual void compute_b(double b[3],double p[3]) const = 0;
		/*! Computes tilde normal */
		virtual void compute_ntil(double ntil[3],double p[3]) const = 0;
		/*! Computes tilde binormal */
		virtual void compute_btil(double btil[3],double p[3]) const = 0;
		/*! Computes the derivative of normal */
		virtual void compute_np(double np[3][3],double p[3]) const = 0;
		/*! Computes the derivative of binormal */
		virtual void compute_bp(double bp[3][3],double p[3]) const = 0;
};

/*******************************************************************/

/*! Kardan parametrisation for AMVis */
class AMVisKardan {
	public:
		/*! Constructor */
		AMVisKardan();
		/*! Destructor */
		virtual ~AMVisKardan();
		
		/*! Computes tangent */
		void compute_t(double t[3],double p[3]) const;
		/*! Computes normal */
		void compute_n(double n[3],double p[3]) const;
		/*! Computes binormal */
		void compute_b(double b[3],double p[3]) const;
		/*! Computes tilde normal */
		void compute_ntil(double ntil[3],double p[3]) const;
		/*! Computes tilde binormal */
		void compute_btil(double btil[3],double p[3]) const;
		/*! Computes the derivative of normal */
		void compute_np(double np[3][3],double p[3]) const;
		/*! Computes the derivative of binormal */
		void compute_bp(double bp[3][3],double p[3]) const;
};

/*******************************************************************/

/*! RevKardan parametrisation for AMVis */
class AMVisRevKardan {
	public:
		/*! Constructor */
		AMVisRevKardan();
		/*! Destructor */
		virtual ~AMVisRevKardan();
		
		/*! Computes tangent */
		void compute_t(double t[3],double p[3]) const;
		/*! Computes normal */
		void compute_n(double n[3],double p[3]) const;
		/*! Computes binormal */
		void compute_b(double b[3],double p[3]) const;
		/*! Computes tilde normal */
		void compute_ntil(double ntil[3],double p[3]) const;
		/*! Computes tilde binormal */
		void compute_btil(double btil[3],double p[3]) const;
		/*! Computes the derivative of normal */
		void compute_np(double np[3][3],double p[3]) const;
		/*! Computes the derivative of binormal */
		void compute_bp(double bp[3][3],double p[3]) const;
};

#endif /* _ELASTIC_BODY_1S_33_RCM_H_ */
