/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>
#include <fstream>
#include <iomanip>
using namespace std;



#include <elastic1sbta.h>

// Constructor
ElasticBody1sBTA::ElasticBody1sBTA(char* body_name, char *pos_file) : ElasticBody1s(body_name, pos_file) {
  int result;
  char* charstringline = new char [charlength];

/*
  Parameter im File: ---------------
  # Class Name
  ElasticBody1sBTA
  
  # Instances
  1
  
  # ASCI (0) or Binary (1)
  0
  
  # Class Stuff
  # Elements
  2
  0.25
  0.0, 0.0, 0.0
  ----------------------------------
 */
  
  // Klassen-Auslesen beginnt bei Class Stuff

  // Zahl der Elemente
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%i" ,&nElements);

  // Elementanzahl
  dofs = ( 5*(nElements +1)    );
  binaryrecordsize=sizeof(float)*(dofs + 1); // Zeit nicht vergessen
  q = new float[dofs];

  // Elementlaenge
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&l0);

  // WrN00
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&WrN00[0],&WrN00[1],&WrN00[2]);

  // J
  parsenextdataline(datafile); // erste Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&J[0][0],&J[0][1],&J[0][2]);
  parsenextdataline(datafile); // zweite Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&J[1][0],&J[1][1],&J[1][2]);
  parsenextdataline(datafile); // dritte Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&J[2][0],&J[2][1],&J[2][2]);

  l0h2 = l0  *l0;
  l0h4 = l0h2*l0h2;

  lGes = nElements*l0;
  lOffSet = lGes * 0.5e-5;
  lGesEff = lGes - 2*lOffSet;

  parsenextdataline(datafile); // Zylinderradius
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&r);

  nContourPoints = 33;
  
  cont_normal = new SVec[nContourPoints];
  contour     = new SVec[nContourPoints];
  
  for(int i=0; i<nContourPoints; i++) {
      cont_normal[i][0] = cos(2*M_PI/(nContourPoints-1)*i);
      cont_normal[i][1] = sin(2*M_PI/(nContourPoints-1)*i);
      contour[i][0]     = r*cont_normal[i][0];
      contour[i][1]     = r*cont_normal[i][1];
  }
  up[1] = 1.0;
}

// Destructor
ElasticBody1sBTA::~ElasticBody1sBTA() {
}

void ElasticBody1sBTA::computeSweep(const int npoints, gleDouble point_array[][3], gleDouble twist_array[]) {
    double ds = lGesEff/(npoints-1);
    
    GLdouble XTemp[6];

    // StartPoint, not drawn
    LocateStructure(XTemp,0.); 
    point_array[0][0] = XTemp[0];
    point_array[0][1] = XTemp[1];
    point_array[0][2] = XTemp[2];
    twist_array[0]    = XTemp[3];
  
    for(int i = 0; i < npoints ; i++) {
	LocateStructure(XTemp,lOffSet + ds*i); 
	point_array[i+1][0] = XTemp[0];
	point_array[i+1][1] = XTemp[1];
	point_array[i+1][2] = XTemp[2];
	twist_array[i+1]    = XTemp[3];
    }

    // StartPoint, not drawn
    LocateStructure(XTemp,lGes); 
    point_array[npoints+1][0] = XTemp[0];
    point_array[npoints+1][1] = XTemp[1];
    point_array[npoints+1][2] = XTemp[2];
    twist_array[npoints+1]    = XTemp[3];

}


// Initialize (Call ElasticBody1s::init()!)
void ElasticBody1sBTA::init() {
  ElasticBody1s::init();
}

/*
// For move camera with body (rigid-body movement)
void ElasticBody1s::ret_rot(float *rotation) {
}
*/

/*
// For move camera with body (rigid-body movement)
void ElasticBody1s::ret_trans(float *translation) {
}
*/


// Text in the left
void ElasticBody1sBTA::return_info_string(QString &info) {
    info=tr("Class: %1\nBody: %2\nPos-File: %3\nBeam\nElemets: %4\nl_0: %5").arg(class_name).arg(body_file_name).arg(posfile_name).arg(nElements).arg(l0);
}







//---------------------------------------------------------------------------
void ElasticBody1sBTA::LocateStructure(GLdouble *X, const double s)
{
    // das richtige Element auswhlen ...
    int nElement = 0;

    while( (nElement+1)*l0 <= s)
	nElement++;

    double sElement = s - nElement * l0; // ElementKOS in der Elementmitte gilt s==0

    if(nElements <= nElement           ) {
	nElement  -= 1;
	sElement +=l0;
    }

    for(int i=0;i<10;i++)
	qElement[i] = q[5*nElement+i]; 

    float &a1 = qElement[0];
    float &w1 = qElement[1];     float &b1 = qElement[2];
    float &u1 = qElement[3];     float &g1 = qElement[4];
    float &a2 = qElement[5];
    float &w2 = qElement[6];     float &b2 = qElement[7];
    float &u2 = qElement[8];     float &g2 = qElement[9];

    static GLdouble Xlocal[6];

    Xlocal[0] = s;
    Xlocal[1] = (b1*sElement + w1 - (Power(sElement,2)*(2*b1*l0 + b2*l0 + 3*w1 - 3*w2))/Power(l0,2) + (Power(sElement,3)*(b1*l0 + b2*l0 + 2*w1 - 2*w2))/Power(l0,3))*cos((a1*(l0 - sElement))/l0 + (a2*sElement)/l0) - (g1*sElement + u1 - (Power(sElement,2)*(2*g1*l0 + g2*l0 + 3*u1 - 3*u2))/Power(l0,2) + (Power(sElement,3)*(g1*l0 + g2*l0 + 2*u1 - 2*u2))/Power(l0,3))*sin((a1*(l0 - sElement))/l0 + (a2*sElement)/l0);
    Xlocal[2] = (g1*sElement + u1 - (Power(sElement,2)*(2*g1*l0 + g2*l0 + 3*u1 - 3*u2))/Power(l0,2) + (Power(sElement,3)*(g1*l0 + g2*l0 + 2*u1 - 2*u2))/Power(l0,3))*cos((a1*(l0 - sElement))/l0 + (a2*sElement)/l0) + (b1*sElement + w1 - (Power(sElement,2)*(2*b1*l0 + b2*l0 + 3*w1 - 3*w2))/Power(l0,2) + (Power(sElement,3)*(b1*l0 + b2*l0 + 2*w1 - 2*w2))/Power(l0,3))*sin((a1*(l0 - sElement))/l0 + (a2*sElement)/l0);
    Xlocal[3] = (a1*(l0 - sElement))/l0 + (a2*sElement)/l0;
//     Xlocal[4] = g1 - (2*sElement*(2*g1*l0 + g2*l0 + 3*u1 - 3*u2))/Power(l0,2) + (3*Power(sElement,2)*(g1*l0 + g2*l0 + 2*u1 - 2*u2))/Power(l0,3);
//     Xlocal[5] = b1 - (2*sElement*(2*b1*l0 + b2*l0 + 3*w1 - 3*w2))/Power(l0,2) + (3*Power(sElement,2)*(b1*l0 + b2*l0 + 2*w1 - 2*w2))/Power(l0,3);

    X[0] = WrN00[0] + J[0][0]*Xlocal[0] + J[0][1]*Xlocal[1] + J[0][2]*Xlocal[2];
    X[1] = WrN00[1] + J[1][0]*Xlocal[0] + J[1][1]*Xlocal[1] + J[1][2]*Xlocal[2];
    X[2] = WrN00[2] + J[2][0]*Xlocal[0] + J[2][1]*Xlocal[1] + J[2][2]*Xlocal[2];

//     X[3] =            J[2][0]*Xlocal[3] + J[2][1]*Xlocal[4] + J[2][2]*Xlocal[5];
//     X[4] =            J[2][0]*Xlocal[3] + J[2][1]*Xlocal[4] + J[2][2]*Xlocal[5];
//     X[5] =            J[2][0]*Xlocal[3] + J[2][1]*Xlocal[4] + J[2][2]*Xlocal[5];

    X[3] =           - Xlocal[3]  *  180./M_PI; // Achtung: GLE will Grad nicht Radiant
}


//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
double ElasticBody1sBTA::Power(double base, int exponent)
{
    return pow(base,exponent);
}

