/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <GL/gl.h>
#include "elastic2s13disk.h" // class's header file

using namespace std;

const int ElasticBody2s13Disk::NodeDofs = 3;
const int ElasticBody2s13Disk::RefDofs  = 2;

// class constructor
ElasticBody2s13Disk::ElasticBody2s13Disk(char* body_name, char * pos_file): ElasticBody( body_name,  pos_file),
  drawPointElement_r(6), drawPointElement_j(72)
{
  int result;
  char* charstringline = new char [charlength];

/*
# Created by AMVisC++Interface
ElasticBody2s13Disk
# number of instances
1
# pos-file is binary=1 (datatype float) or ascii=0
0
# ---------------- END INPUT OF AMVis::Elastic and higher
# Color
0
# nr
1
# nj
4
# Elements
4
# Nodes
8
# Ri Ra
0.1 0.33
# di
 0.03 0 0
# da
 0.03 0 0
# WrON00
0 0 0
# J
1 0 0 
0 1 0 
0 0 1 
# NodeCoordinates
0.1 0
0.1 1.5708
0.1 3.14159
0.1 4.71239
0.33 0
0.33 1.5708
0.33 3.14159
0.33 4.71239
# ElementNodeList
0 1 4 5
1 2 5 6
2 3 6 7
3 0 7 4
*/
  // Farbe
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f" ,&colorvalue);
  if(0.0 <= colorvalue && colorvalue <= 1.0)  translateColorvalueToRGBA(colorvalue, &color[0], &color[1], &color[2], &color[3]);
  else {
	color[0] = 0.65;
	color[1] = 0.65;
	color[2] = 1.25;
	color[3] = 1.25;
  }
  // Zahl der Elemente radial
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%i" ,&nr);
  drawPointElement_r = max(3,drawPointElement_r/nr); // mindestens 4 stuetzpunkte pro element
  // Zahl der Elemente azimutal
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%i" ,&nj);
  drawPointElement_j = max(4,drawPointElement_j/nj); // mindestens 4 stuetzpunkte pro element

  delta_r = 1./double(drawPointElement_r);
  delta_j = 1./double(drawPointElement_j);

  // Zahl der Elemente gesamt
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%i" ,&Elements);
  ENL = new ElementNodeList[Elements];
  // Zahl der Konten gesamt
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%i" ,&Nodes);
  NC = new NodeCoordinates[Nodes];

  dofs = ( RefDofs + NodeDofs*Nodes );
  binaryrecordsize=sizeof(float)*(dofs + 1); // Zeit nicht vergessen
  q = new float[dofs];

  // Radien innen, aussen
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f" ,&Ri,&Ra);
  // dicke innen
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&di[0],&di[1],&di[2]);
  // dicke aussen
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&da[0],&da[1],&da[2]);

  // WrN00
  parsenextdataline(datafile);
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&WrN00[0],&WrN00[1],&WrN00[2]);

/////////////////////  // Jacobi-Matrix
/////////////////////  parsenextdataline(datafile); // erste Zeile
/////////////////////  result=GetCharacterLine(charstringline, charlength, datafile);
/////////////////////  sscanf(charstringline, "%f %f %f" ,&J[0][0],&J[0][1],&J[0][2]);J[0][3]=0;
/////////////////////  parsenextdataline(datafile); // zweite Zeile
/////////////////////  result=GetCharacterLine(charstringline, charlength, datafile);
/////////////////////  sscanf(charstringline, "%f %f %f" ,&J[1][0],&J[1][1],&J[1][2]);J[1][3]=0;
/////////////////////  parsenextdataline(datafile); // dritte Zeile
/////////////////////  result=GetCharacterLine(charstringline, charlength, datafile);
/////////////////////  sscanf(charstringline, "%f %f %f" ,&J[2][0],&J[2][1],&J[2][2]);J[2][3]=0;
/////////////////////  J[3][0] = 0; J[3][1] = 0; J[3][2] = 0; J[3][3] = 1;
/////////////////////
/////////////////////  Axis[0] = J[0][0]; Axis[1] = J[1][0]; Axis[2] = J[2][0]; // Achse ist erste Spalte J

  J = new GLfloat[4*4];
  // Jacobi-Matrix
  parsenextdataline(datafile); // erste Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&J[0],    &J[4+0],    &J[8+0]);    J[12+0]=0;
  parsenextdataline(datafile); // zweite Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&J[1],    &J[4+1],    &J[8+1]);    J[12+1]=0;
  parsenextdataline(datafile); // dritte Zeile
  result=GetCharacterLine(charstringline, charlength, datafile);
  sscanf(charstringline, "%f %f %f" ,&J[2],    &J[4+2],    &J[8+2]);    J[12+2]=0;
                                      J[3] = 0; J[4+3] = 0; J[8+3] = 0; J[12+3]=1;

  Axis[0] = J[0]; Axis[1] = J[1]; Axis[2] = J[2]; // Achse ist erste Spalte J

  // NodeCoordinates
  for(int i=0;i<Nodes;i++) {
    parsenextdataline(datafile); // erste Zeile
    result=GetCharacterLine(charstringline, charlength, datafile);
    sscanf(charstringline, "%f %f" ,&NC[i].r,&NC[i].j);
  }
 
  // ElementNodeList
  for(int i=0;i<Nodes;i++) {
    parsenextdataline(datafile); // erste Zeile
    result=GetCharacterLine(charstringline, charlength, datafile);
    sscanf(charstringline, "%i %i %i %i" ,&ENL[i].node0,&ENL[i].node1,&ENL[i].node2,&ENL[i].node3);
  }
}

ElasticBody2s13Disk::~ElasticBody2s13Disk()
{
}

// Text in the left
void ElasticBody2s13Disk::return_info_string(QString &info) {
    info=tr("Class: %1\nBody: %2\nPos-File: %3\nElemets(r,j): %4,%5\nAxis: %6,%7,%8").arg(class_name).arg(body_file_name).arg(posfile_name).arg(nr).arg(nj).
arg(Axis[0]).arg(Axis[1]).arg(Axis[2]);
}

// For move camera with body (rigid-body movement)
void ElasticBody2s13Disk::ret_rot(float *rotation) {
    rotation[0] = Axis[0]*q[1];
    rotation[1] = Axis[1]*q[1];
    rotation[2] = Axis[2]*q[1];
}



// For move camera with body (rigid-body movement)
void ElasticBody2s13Disk::ret_trans(float *translation) {
    translation[0] = Axis[0]*q[0];
    translation[1] = Axis[1]*q[0];
    translation[2] = Axis[2]*q[0];
}

void ElasticBody2s13Disk::init()
{
    ElasticBody::init();
    for(int i=0;i<dofs;i++)
	  q[0] = 0.0;

//	cout  << nr <<" , " << nj <<endl;
//	cout  << Ri <<" , " << Ra <<endl;
//	cout  << Axis[0] <<" , " << Axis[1] <<" , " << Axis[2] <<endl;
//	for(int i=0;i<Nodes;i++) cout <<"Node" << i <<":" << NC[i].r <<" , " << NC[i].j <<endl;
//	for(int i=0;i<Elements;i++) cout <<"Element" << i <<":" << ENL[i].node0 <<" , " << ENL[i].node1 <<" , " << ENL[i].node2 <<" , " << ENL[i].node3 <<endl;

 }

void ElasticBody2s13Disk::draw()
{
    ElasticBody::draw();

    // Farben
    glEnable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
    glColorMaterial (GL_FRONT_AND_BACK, GL_DIFFUSE);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
    static GLfloat mat_specular[] = {1.0,1.0,1.0, 1.0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
    static GLfloat color_shininess = 20;
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
    float em[]={0,0,0,0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);
    glEnable(GL_SMOOTH);
///   // Farben
///    glEnable(GL_COLOR_MATERIAL);
///    glColorMaterial (GL_FRONT_AND_BACK, GL_DIFFUSE);
///    GLfloat color[] = {1.2,2.2,1.2,1.8};
///    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
///    GLfloat mat_specular[] = {1.25,1.9,1.25, 1.1};
///    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
///    GLfloat color_shininess = 12;
///    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
///    float em[4]={0,0,0,0};
///    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);

    glPushMatrix();
    // Aufpunktbewegung
	glMultMatrixf(J);
    static const double radTOdeg = 180./M_PI;
	glTranslatef( q[0], 0, 0  );             // lokale x-Achse
    glRotatef   ( q[1]*radTOdeg, 1 , 0, 0 ); // lokale x-Achse

    drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!

    if(contour_flag==true) {
	// draw contour
	glEnable(GL_POLYGON_OFFSET_LINE);
	glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	glLineWidth(1.0);
	glPolygonOffset(-1.0,0.0);
	glDisable(GL_LIGHTING);
	
	drawBodyOutline();
    }
    
    if(selected_flag==true) {
	// draw something
    }
    
    if(local_cos_flag==true) {
	// draw something
    }
    glPopMatrix();
}

void ElasticBody2s13Disk::drawBody() {

  glLineWidth(1.0);
  for(int i=0;i<Elements;i++) {
//  for(int i=0;i<1;i+=1) {
	NodeRJ0[0] = NC[ENL[i].node0].r; NodeRJ0[1] = NC[ENL[i].node0].j;
	NodeRJ1[0] = NC[ENL[i].node1].r; NodeRJ1[1] = NC[ENL[i].node1].j;
    if(NodeRJ1[1]<NodeRJ0[1]) NodeRJ1[1] +=2*M_PI;
	NodeRJ2[0] = NC[ENL[i].node2].r; NodeRJ2[1] = NC[ENL[i].node2].j;
//  if(NodeRJ2[1]<NodeRJ0[1]) NodeRJ2[1] +=2*M_PI;
	NodeRJ3[0] = NC[ENL[i].node3].r; NodeRJ3[1] = NC[ENL[i].node3].j;
    if(NodeRJ3[1]<NodeRJ0[1]) NodeRJ3[1] +=2*M_PI;

//	glNormal3dv(normal(0.5*Delta,0.5));
//	  if(1<x)
//		glNormal3dv(normal((x-0.5)*Delta,0.5));	

    // azimutale Interpunktion	
	for (int j=0;j<drawPointElement_j;j++) {
	  glBegin( GL_QUAD_STRIP );
	  for (int r=0;r<=drawPointElement_r;r++) {
		// radiale Interpunktion	
		glVertex3dv(position(r*delta_r,(j  )*delta_j,i));
		glVertex3dv(position(r*delta_r,(j+1)*delta_j,i));
	  }
	  glEnd();
	}
  }
}

void ElasticBody2s13Disk::drawBodyOutline() {
  glEnable(GL_LINE_SMOOTH);
  glColor3f(0.0, 0.0, 0.0);
  glLineWidth(1.5);

  for(int i=0;i<Elements;i++) {
	NodeRJ0[0] = NC[ENL[i].node0].r; NodeRJ0[1] = NC[ENL[i].node0].j;
	NodeRJ1[0] = NC[ENL[i].node1].r; NodeRJ1[1] = NC[ENL[i].node1].j;
	if(NodeRJ1[1]<NodeRJ0[1]) NodeRJ1[1] +=2*M_PI;
	NodeRJ2[0] = NC[ENL[i].node2].r; NodeRJ2[1] = NC[ENL[i].node2].j;
	//  if(NodeRJ2[1]<NodeRJ0[1]) NodeRJ2[1] +=2*M_PI;
	NodeRJ3[0] = NC[ENL[i].node3].r; NodeRJ3[1] = NC[ENL[i].node3].j;
	if(NodeRJ3[1]<NodeRJ0[1]) NodeRJ3[1] +=2*M_PI;

	glBegin( GL_LINES );
	glVertex3dv(position(0,0,i));    glVertex3dv(position(1,0,i));
	// azimutale Interpunktion	
	for (int j=0;j<drawPointElement_j;j++) {
	  glVertex3dv(position(1,(j  )*delta_j,i));
	  glVertex3dv(position(1,(j+1)*delta_j,i));
	}
	glVertex3dv(position(1,1,i));    glVertex3dv(position(0,1,i));
	// azimutale Interpunktion	
	for (int j=drawPointElement_j;0<j;j--) {
	  glVertex3dv(position(1,(j  )*delta_j,i));
	  glVertex3dv(position(1,(j-1)*delta_j,i));
	}
	glEnd();
  }
}


GLdouble* ElasticBody2s13Disk::position(const double &R, const double &Phi, int Element) {
    GLdouble *X = new GLdouble[3];
    double r = (1-R)*(1-Phi)*NodeRJ0[0]
	         + (1-R)*(  Phi)*NodeRJ1[0]
			 + (  R)*(  Phi)*NodeRJ3[0]
			 + (  R)*(1-Phi)*NodeRJ2[0];
    double j = (1-R)*(1-Phi)*NodeRJ0[1]
	         + (1-R)*(  Phi)*NodeRJ1[1]
			 + (  R)*(  Phi)*NodeRJ3[1]
			 + (  R)*(1-Phi)*NodeRJ2[1];
	X[1] = cos(j) * r;
	X[2] = sin(j) * r;
    X[0] = (1-R)*(1-Phi)*q[RefDofs + NodeDofs*ENL[Element].node0]
	     + (1-R)*(  Phi)*q[RefDofs + NodeDofs*ENL[Element].node1]
		 + (  R)*(  Phi)*q[RefDofs + NodeDofs*ENL[Element].node3]
		 + (  R)*(1-Phi)*q[RefDofs + NodeDofs*ENL[Element].node2];

    return X;
}

/*GLdouble* ElasticBody2s13Disk::normal(const double &x, const double &y) {
    GLdouble *X = new GLdouble[3];
    const double &x1 = q[0]; const double &y1 = q[ 1]; const double &z1 = q[ 2];
    const double &x2 = q[3]; const double &y2 = q[ 4]; const double &z2 = q[ 5];
    const double &x3 = q[6]; const double &y3 = q[ 7]; const double &z3 = q[ 8];
    const double &x4 = q[9]; const double &y4 = q[10]; const double &z4 = q[11];

    X[0] = -( -y2*z1 + y4*z1 + y1*z2 - y4*z2 - y1*z4 + y2*z4 + 
	x*(y3*z1 - y4*z1 - y3*z2 + y4*z2 - y1*z3 + y2*z3 + y1*z4 - y2*z4) + 
	y*(y2*z1 - y3*z1 - y1*z2 + y4*z2 + y1*z3 - y4*z3 - y2*z4 + y3*z4) );

    X[1] = -( x2*z1 - x4*z1 - x1*z2 + x4*z2 + x1*z4 - x2*z4 + 
	y*(-x2*z1 + x3*z1 + x1*z2 - x4*z2 - x1*z3 + x4*z3 + x2*z4 - x3*z4) + 
	x*(x4*(z1 - z2) + x3*(-z1 + z2) + x1*(z3 - z4) + x2*(-z3 + z4)) );

    X[2] = -( -x2*y1 + x4*y1 + x1*y2 - x4*y2 - x1*y4 + x2*y4 + 
	x*(x3*y1 - x4*y1 - x3*y2 + x4*y2 - x1*y3 + x2*y3 + x1*y4 - x2*y4) + 
	y*(x2*y1 - x3*y1 - x1*y2 + x4*y2 + x1*y3 - x4*y3 - x2*y4 + x3*y4) );

    double nrm = sqrt( X[0]*X[0] + X[1]*X[1] + X[2]*X[2] );
    X[0] /= nrm;    X[1] /= nrm;    X[2] /= nrm; 
    
    return X;
}*/

void ElasticBody2s13Disk::draw_pure() {
    ElasticBody::draw_pure();

    glEnable(GL_SMOOTH);


    GLdouble p[4][3];
    for (int j=0;j<4;j++)
	for (int i=0;i<3;i++) p[j][i] = q[3*j+i];
    glBegin( GL_QUADS );
    glVertex3dv(p[0]);  glVertex3dv(p[1]);  glVertex3dv(p[2]);  glVertex3dv(p[3]);
    glEnd();
}
