/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <GL/gl.h>
#include "glwidget.h"
#include <QInputDialog>

using namespace std;
 
double         ViewMatrix[16];
double        wx0, wy0, scale;
GLuint         object;
int side;
GLint un_dis_view[4];
static GLdouble saveState[16] = {1.0,0.0,0.0,0.0,
   0.0,1.0,0.0,0.0,
   0.0,0.0,1.0,0.0,
   0.0,0.0,0.0,1.0};
static GLfloat followState[16] = {1.0,0.0,0.0,0.0,
   0.0,1.0,0.0,0.0,
   0.0,0.0,1.0,0.0,
   0.0,0.0,0.0,1.0};

int w_width;
int w_height;
int u;
int linestart = 10;                 
int linespace = 20;


void GLWidget::HelpDisplay(GLint ww, GLint wh)
{
   glDisable(GL_LIGHTING);
   glDisable(GL_TEXTURE_2D);
   if(bgBlack->isChecked())
     glColor3f(1.0, 1.0, 1.0);
   else if(bgWhite->isChecked())
     glColor3f(0.0, 0.0, 0.0);
   else if(bgColor->isChecked())
     glColor3f(0.0f,0.0f,0.0f);
   glViewport( 0, 0, w_width, w_height);
   QString DisplayString;
   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   gluOrtho2D(0, ww, 0, wh);
   glScalef(1, -1, 1);
   glTranslatef(0, -wh, 0);
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glLoadIdentity();
   linestart = 10;
   DisplayString=tr("Current time: %1s").arg(NewGLContainer.object_time());
   renderText(10, 20, DisplayString);
   if(bgBlack->isChecked())
     glColor3f(1.0, 1.0, 1.0);
   else if(bgWhite->isChecked())
     glColor3f(0.0, 0.0, 0.0);
   else if(bgColor->isChecked())
     glColor3f(1,1,1);
   renderText(10, wh-10, tr("Created by AMVis; Institute of Applied Mechanics, TUM"));
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW);
   glEnable(GL_LIGHTING);
//   glEnable(GL_TEXTURE_2D);
   side = qMax(w_width, w_height);
   glViewport((w_width - side) / 2, (w_height - side) / 2, side, side);
}

GLWidget::GLWidget(QWidget *parent)
   : QGLWidget(parent), lastfps(0)
{
   setFocusPolicy(Qt::StrongFocus); // change to NoFocus FW
   setMouseTracking(true);
   setCursor(Qt::CrossCursor);
   lastdatapos_timer = new QTimer(this);
   slideshow_timer = new QTimer(this);
   connect(lastdatapos_timer, SIGNAL(timeout()), this, SLOT(lastdatapos_timer_slot()));
   connect(slideshow_timer, SIGNAL(timeout()), this, SLOT(slideshow_timer_slot()));
   lastdatapos_timer_running = false ;
   slideshow_timer_running = false;
   follow_object = false;
   lastr = 0;
   slideshow_time=1;
   lastpos_record = -1;
   fps=new QLabel("");

   NewGLContainer.setGLWidget(this);
}

GLWidget::~GLWidget()
{

}


void GLWidget::KardanMatrixInvert(float* angles, float  out[16])
{
   out[0]  = cos (angles[1])  * cos (angles[2]) ;
   out[4]  = cos (angles[0]) * sin (angles[2]) + sin (angles[0]) * sin (angles[1]) * cos(angles[2]);  ;
   out[8]  = sin (angles[0]) * sin (angles[2]) - cos (angles[0]) * sin (angles[1])*cos(angles[2]) ;
   out[12]  = 0.0 ;
   out[1]  =-cos (angles[1])  * sin (angles[2]) ;
   out[5]  = cos (angles[0]) * cos (angles[2]) - sin (angles[0]) * sin (angles[1]) * sin (angles[2]) ;
   out[9]  = sin (angles[0]) * cos (angles[2]) + cos(angles[0]) * sin (angles[1]) * sin (angles[2]) ;
   out[13]  = 0.0 ;
   out[2]  = sin (angles[1]) ;
   out[6]  =-sin (angles[0]) * cos (angles[1]);
   out[10] = cos (angles[0]) * cos (angles[1]);
   out[14] = 0.0 ;
   out[3] = 0.0 ;
   out[7] = 0.0 ;
   out[11] = 0.0 ;
   out[15] = 1.0 ;
}


void GLWidget::initializeGL()
{
   glEnable( GL_DEPTH_TEST);   
   glDepthFunc( GL_LESS);
   glShadeModel( GL_SMOOTH);   
   glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, 1);
   glMatrixMode( GL_PROJECTION);
   glLoadIdentity();
   glMatrixMode( GL_MODELVIEW);
   glLoadIdentity();
   glClearColor(0.0, 0.0, 0.0, 0.0);
   glEnable(GL_DEPTH_TEST);
   glShadeModel(GL_SMOOTH);
   glClearColor( 0.0, 0.0, 0.0, 1.0);
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   glPushMatrix();
   glLoadIdentity();
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   glEnable(GL_NORMALIZE);
   NewGLContainer.Create();

   // draw environment
   vector<CBody*>::iterator i;
   for(i=NewGLContainer.BodyEnv.begin(); i!=NewGLContainer.BodyEnv.end(); i++) {
     (*i)->init();
   }

   StatusBar->addPermanentWidget(fps);
   fpstimer.start();
}

void GLWidget::paintGL()
{
   show_object_info();
   glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
   float invert[16];
   double scale_factor;
   float positions[3];
   float rotations[3];
   float minimum_x;
   float maximum_x;
   float minimum_y;
   float maximum_y;
   float minimum_z;
   float maximum_z;

   GLfloat ambient[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat diffuse[] = { 0.60, 0.6, 0.6, 1.0 };
   GLfloat specular[] = { 0.15, 0.15, 0.15, 1.0 };
   GLfloat position[] = { -5.5, 5.5, 1.00, 0.0 };

   GLfloat ambient3[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat diffuse3[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat specular3[] = { 0.8, 0.8, 0.8, 1.0  };
   GLfloat position3[] = {1.0, 1.0, -0.02, 0.0};

   GLfloat ambient4[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat diffuse4[] = { 0.45, 0.45, 0.45, 1.0 };
   GLfloat specular4[] = {  0.15, 0.15, 0.15, 1.0 };
   GLfloat position4[] = {- 0.5, 0.5, 1.00, 0.0};

   if (follow_object == false)
   {
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      glMultMatrixd( saveState );
   } else
   {
      glMatrixMode( GL_MODELVIEW);
      glPushMatrix();
      glLoadIdentity();
      NewGLContainer.get_current_trans(positions);
      NewGLContainer.get_current_rot(rotations);
      NewGLContainer.get_object_dimensions(& minimum_x,& maximum_x, & minimum_y, & maximum_y,& minimum_z, & maximum_z);
      KardanMatrixInvert(rotations,invert);
      glMultMatrixf(invert);
      scale_factor =  0.8 *0.150 / (maximum_x);
      glScaled(scale_factor, scale_factor, scale_factor);
      glTranslatef(-positions[0],-positions[1], -positions[2]);
      glGetFloatv( GL_MODELVIEW_MATRIX, followState );
      glPopMatrix();
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      glMultMatrixd( saveState );
      glMultMatrixf( followState );
   };
   glPushMatrix();
   glLoadIdentity();
   glLightfv(GL_LIGHT1, GL_POSITION, position);
   glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
   glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
   glLightfv(GL_LIGHT1, GL_SPECULAR, specular);
   glEnable(GL_LIGHT1);
   glPopMatrix();

   glPushMatrix();
   glLightfv(GL_LIGHT3, GL_POSITION, position3);
   glLightfv(GL_LIGHT3, GL_AMBIENT, ambient3);
   glLightfv(GL_LIGHT3, GL_DIFFUSE, diffuse3);
   glLightfv(GL_LIGHT3, GL_SPECULAR, specular3);
   glEnable(GL_LIGHT3);
   glPopMatrix();

   glPushMatrix();
   glLoadIdentity();
   glLightfv(GL_LIGHT4, GL_AMBIENT, ambient4);
   glLightfv(GL_LIGHT4, GL_DIFFUSE, diffuse4);
   glLightfv(GL_LIGHT4, GL_SPECULAR, specular4);
   glLightfv(GL_LIGHT4, GL_POSITION, position4);
   glEnable(GL_LIGHT4);
   glPopMatrix(); 
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   draw_background();
   glPopMatrix();

   double x1,y1,z1,x2,y2,z2;
   double projMat[16];
   int viewport[4];
   glGetDoublev(GL_PROJECTION_MATRIX, projMat);
   glGetIntegerv(GL_VIEWPORT, viewport);
   double dummy[16];
   if(follow_object==false)
     memcpy(dummy,saveState,sizeof(double)*16);
   else {
     glMatrixMode(GL_MODELVIEW);
     glPushMatrix();
     glLoadMatrixd(saveState);
     glMultMatrixf(followState);
     glGetDoublev(GL_MODELVIEW_MATRIX, dummy);
     glPopMatrix();
   }
   gluUnProject(0,0,0.5,dummy,projMat,viewport,&x1,&y1,&z1);
   gluUnProject(width(),0,0.5,dummy,projMat,viewport,&x2,&y2,&z2);
   float length=sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1)+(z2-z1)*(z2-z1));
   if(drawWorldCosAct->isChecked()) {
     glEnable(GL_COLOR_MATERIAL);
     glDisable(GL_LIGHTING);
     glEnable(GL_LINE_SMOOTH);
     glLineWidth(3.0);
     glEnable(GL_LINE_STIPPLE);
     glLineStipple(2,0xEEEE); // 0xEEEE = bin 1110111011101110
     // X DIR red
     glColor3ub(255, 100, 100);
     glBegin(GL_LINE_STRIP);
     glVertex3f(0.0, 0.0, 0.0);
     glVertex3f(length/10, 0.0, 0.0);
     glEnd();
     // Y DIR green
     glColor3ub(100, 255, 100);
     glBegin(GL_LINE_STRIP);
     glVertex3f(0.0, 0.0, 0.0);
     glVertex3f(0.0, length/10, 0.0);
     glEnd();
     // Z DOR blue
     glColor3ub(100, 100, 255);
     glBegin(GL_LINE_STRIP);
     glVertex3f(0.0, 0.0, 0.0);
     glVertex3f(0.0, 0.0, length/10);
     glEnd();
     glEnable(GL_LIGHTING);
     glDisable(GL_COLOR_MATERIAL);
     glDisable(GL_LINE_STIPPLE);
   }
   glEnable(GL_COLOR_MATERIAL);
   if(drawXYPlaneAct->isChecked()) {
     glBegin(GL_QUADS);
       glNormal3f(0, 0, 1);
       glColor3f(0,0,1);
       glVertex3f(-length*2,-length*2,0);
       glVertex3f(+length*2,-length*2,0);
       glVertex3f(+length*2,+length*2,0);
       glVertex3f(-length*2,+length*2,0);
     glEnd();
   }
   if(drawXZPlaneAct->isChecked()) {
     glBegin(GL_QUADS);
       glNormal3f(0, 1, 0);
       glColor3f(0,1,0);
       glVertex3f(-length*2,0,-length*2);
       glVertex3f(+length*2,0,-length*2);
       glVertex3f(+length*2,0,+length*2);
       glVertex3f(-length*2,0,+length*2);
     glEnd();
   }
   if(drawYZPlaneAct->isChecked()) {
     glBegin(GL_QUADS);
       glNormal3f(1, 0, 0);
       glColor3f(1,0,0);
       glVertex3f(0,-length*2,-length*2);
       glVertex3f(0,+length*2,-length*2);
       glVertex3f(0,+length*2,+length*2);
       glVertex3f(0,-length*2,+length*2);
     glEnd();
   }
   glDisable(GL_COLOR_MATERIAL);

   NewGLContainer.draw_all_objects();

   // draw environment
   vector<CBody*>::iterator i;
   for(i=NewGLContainer.BodyEnv.begin(); i!=NewGLContainer.BodyEnv.end(); i++) {
     (*i)->contour_flag=true;
     (*i)->draw();
   }

   glPopMatrix();
   HelpDisplay(w_width, w_height);
   glFlush();

   // Output PT1 filtered frames per second rate
   float T=0.1; // Time constant
   float dt=fpstimer.restart()/1000.0;
   float curfps=1/dt;
   lastfps=(curfps+T/dt*lastfps)/(T/dt+1);
   //fps->setText(tr("FPS: %1").arg(lastfps,0,'f',1));
   fps->setText(tr("FPS: %1").arg(lastfps,0,'f',1));
}

void GLWidget::resizeGL(int width, int height)
{
   side = qMax(width, height);
   glViewport((width - side) / 2, (height - side) / 2, side, side);
   w_width=width;
   w_height=height;
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glOrtho(-0.150,0.150,-0.150,0.150,-20.0, 20.0);    
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}

void GLWidget::mousePressEvent(QMouseEvent *event)
{

      lastPos = event->pos();
      wx0 = event->x();
      wy0 = event->y();


   // Bei Click Cursor-Coord auf Konsole ausgeben
   static double xpre=0, ypre=0, zpre=0;
   double x,y,z;
   double projMat[16];
   int viewport[4];
   glGetDoublev(GL_PROJECTION_MATRIX, projMat);
   glGetIntegerv(GL_VIEWPORT, viewport);
   double dummy[16];
   if(follow_object==false)
     memcpy(dummy,saveState,sizeof(double)*16);
   else {
     glMatrixMode(GL_MODELVIEW);
     glPushMatrix();
     glLoadMatrixd(saveState);
     glMultMatrixf(followState);
     glGetDoublev(GL_MODELVIEW_MATRIX, dummy);
     glPopMatrix();
   }
   gluUnProject(event->x(),w_height-1-event->y(),0.5,dummy,projMat,viewport,&x,&y,&z);
   cout<<endl
       <<"Clicked cursor coord = ["<<x<<"; "<<y<<"; "<<z<<"]"<<endl
       <<"Delta to last point = ["<<x-xpre<<"; "<<y-ypre<<"; "<<z-zpre<<"]"<<endl
       <<"Distance to last point = "<<sqrt((x-xpre)*(x-xpre)+(y-ypre)*(y-ypre)+(z-zpre)*(z-zpre))<<endl;
   xpre=x; ypre=y; zpre=z;


   if((event->modifiers() & Qt::AltModifier) || (event->modifiers() & Qt::ShiftModifier)) {
      updateGL();
      glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
      if (follow_object == false)
      {
	 glMatrixMode(GL_MODELVIEW);
	 glPushMatrix();
	 glMultMatrixd( saveState );
      } else
      {
	 float invert[16];
	 double scale_factor;
	 float positions[3];
	 float rotations[3];
	 float minimum_x;
	 float maximum_x;
	 float minimum_y;
	 float maximum_y;
	 float minimum_z;
	 float maximum_z;
	 glMatrixMode( GL_MODELVIEW);
	 glPushMatrix();
	 glLoadIdentity();
	 NewGLContainer.get_current_trans(positions);
	 NewGLContainer.get_current_rot(rotations);
	 NewGLContainer.get_object_dimensions(& minimum_x,& maximum_x, & minimum_y, & maximum_y,& minimum_z, & maximum_z);
	 KardanMatrixInvert(rotations,invert);
	 glMultMatrixf(invert);
	 scale_factor =  0.8 *0.150 / (maximum_x);
	 glScaled(scale_factor, scale_factor, scale_factor);
	 glTranslatef(-positions[0],-positions[1], -positions[2]);
	 glGetFloatv( GL_MODELVIEW_MATRIX, followState );
	 glPopMatrix();
	 glMatrixMode(GL_MODELVIEW);
	 glPushMatrix();
	 glMultMatrixd( saveState );
	 glMultMatrixf( followState );

      };


      NewGLContainer.draw_all_objects_pure();
      float pixel[3];
      long selected_object;
      QString selec;
      glReadPixels(event->x(), w_height-1-event->y(), 1, 1, GL_RGB, GL_FLOAT, pixel);
      selected_object= NewGLContainer.return_picked_object( pixel);
      glPopMatrix();
      if (selected_object>=0)
      {
	 selec.setNum(selected_object);
	 selec = "Selected object " + selec;
	 StatusBar->showMessage(selec);
	 NewGLContainer.set_current_object(selected_object);
        if(event->modifiers() & Qt::ControlModifier)
          ListBox->setItemSelected(ListBox->item(selected_object), TRUE);
	else {
	  for(int i=0; i<ListBox->count(); i++) ListBox->setItemSelected(ListBox->item(i),false);
          ListBox->setItemSelected(ListBox->item(selected_object), TRUE);
	}
                
        update_object_properties();
        show_object_info();
      }else
      {
	 StatusBar->showMessage("No object selected!");
      };
      updateGL();
   }

   if(event->modifiers() & Qt::ShiftModifier) {
     follow_object_slot();
   }

}

void GLWidget::mouseMoveEvent(QMouseEvent *event)
{
  if(event->buttons()!=Qt::NoButton) {
   if (event->modifiers()==Qt::NoModifier)
   {
      int dx = (event->x() - lastPos.x());
      int dy = (event->y() - lastPos.y());
      int x, y;
      double  rx, ry, dx2, dy2, d1;
      x = event->x();
      y = event->y();
      dx2 = (x - wx0) / w_height;
      dy2 = (y - wy0) / w_height;
      rx = (x - .5 * w_width) / w_height;
      ry = (y - .5 * w_height) / w_height;
      wx0 = x;
      wy0 = y;
      lastPos = event->pos();
      if (event->buttons() & Qt::LeftButton) {
	 xRot= dy;
	 yRot=  dx;
	 glPushMatrix();
	 glLoadIdentity();
	 d1 = sqrt( dx2*dx2 + dy2*dy2);
	 glRotated( 360 * d1/2.5, dy2, dx2, 4 * (ry*dx2 - rx*dy2));
	 glMultMatrixd( saveState );
	 glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
	 glPopMatrix();
      }
      if (event->buttons() & Qt::RightButton) {

	 glPushMatrix();
	 glLoadIdentity();
	 glTranslatef(dx/2500.0, -dy/2500.0, 0.0);
	 glMultMatrixd( saveState );
	 glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
	 glPopMatrix();
      }
      if (event->buttons() & Qt::MidButton) {
	 glPushMatrix();
	 glLoadIdentity();
	 scale = exp((double)dy*0.005);
	 glScaled( scale,scale,scale);
	 glMultMatrixd( saveState );
	 glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
	 glPopMatrix();
      } //middle mouse
//      updateGL();
   }
   else
   {
   };
   lastPos = event->pos();
   updateGL();
  }

  double x,y,z;
  double projMat[16];
  int viewport[4];
  glGetDoublev(GL_PROJECTION_MATRIX, projMat);
  glGetIntegerv(GL_VIEWPORT, viewport);
  double dummy[16];
  if(follow_object==false)
    memcpy(dummy,saveState,sizeof(double)*16);
  else {
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadMatrixd(saveState);
    glMultMatrixf(followState);
    glGetDoublev(GL_MODELVIEW_MATRIX, dummy);
    glPopMatrix();
  }
  gluUnProject(event->x(),w_height-1-event->y(),0.5,dummy,projMat,viewport,&x,&y,&z);
  StatusBar->showMessage(tr("Cursor-Coord.: %1 , %2 , %3").arg(x,0,'f',8).arg(y,0,'f',8).arg(z,0,'f',8));
}

void GLWidget::start_auto_play(bool autoPlay) {
   if(autoPlay) {
	 initializeGL();
	 button_slideshow_current_slot();
   }
}

void GLWidget::set_view_state(GLdouble state[16]) { //, bool& followObject) {
  update();
  updateGL();
  glPushMatrix();
  glLoadMatrixd(state);
  glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
  glPopMatrix();
  updateGL();
}

void GLWidget::get_view_state(GLdouble state[16]) { //, bool& followObject) {
//   glGetDoublev( GL_MODELVIEW_MATRIX, state );
  for(int i=0;i<16;i++)
	state[i]  = saveState[i];
//  followObject = follow_object;
}

void GLWidget::look_from_top()
{
   follow_object = false;
   glPushMatrix();
   glLoadIdentity();
   glRotatef(+90.0, 1.0, 0.0, 0.0);
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   updateGL();
}

void GLWidget::look_from_bottom()
{
   follow_object = false;
   glPushMatrix();
   glLoadIdentity();
   glRotatef(-90.0, 1.0, 0.0, 0.0);
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   updateGL();
}

void GLWidget::look_from_front()
{
   follow_object = false;
   glPushMatrix();
   glLoadIdentity();
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   updateGL();
}

void GLWidget::look_from_back()
{
   follow_object = false;
   glPushMatrix();
   glLoadIdentity();
   glRotatef(+180.0,0.0,1.0,0.0);
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   updateGL();
}

void GLWidget::look_from_left()
{
   follow_object = false;
   glPushMatrix();
   glLoadIdentity();
   glRotatef(+90.0, 0.0, 1.0, 0.0);
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   updateGL();
}

void GLWidget::look_from_right()
{
   follow_object = false;
   glPushMatrix();
   glLoadIdentity();
   glRotatef(-90.0, 0.0, 1.0, 0.0);
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   updateGL();
}

void GLWidget::draw_background(void)
{
   GLdouble model[4*4];
   GLdouble proj[4*4];
   glDisable (GL_TEXTURE_2D);
   GLdouble xlo,ylo,zlo,xru, yru, zru;
   glGetDoublev(GL_MODELVIEW_MATRIX, model);
   glGetDoublev(GL_PROJECTION_MATRIX, proj);
   glViewport( 0, 0, w_width, w_height );
   glGetIntegerv(GL_VIEWPORT, un_dis_view);
   glDisable(GL_CULL_FACE);
   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   if(  gluUnProject(0.0, w_height,1.0,
		     model, proj, un_dis_view,
		     &xlo, &ylo, &zlo)>0)
   {
      gluUnProject(w_width, 0.0,1.0,
		   model, proj, un_dis_view,
		   &xru, &yru, &zru);
      glPushMatrix();
      glDisable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
      glLoadIdentity();
      glShadeModel(GL_SMOOTH);
      glBegin(GL_QUADS);
      if(bgBlack->isChecked())
        glColor3f(0.0, 0.0, 0.0);
      else if(bgWhite->isChecked())
        glColor3f(1.0, 1.0, 1.0);
      else if(bgColor->isChecked())
        glColor3f(0.83, 0.83, 1.0);
      glVertex3f(-0.150, 0.150, -19.0);// Left And Up 1 Unit (Top Left)
      glVertex3f( 0.150, 0.150, -19.0);// Right And Up 1 Unit (Top Right)
      if(bgBlack->isChecked())
        glColor3f(0.0, 0.0, 0.0);
      else if(bgWhite->isChecked())
        glColor3f(1.0, 1.0, 1.0);
      else if(bgColor->isChecked())
        glColor3f(0.35, 0.35, 0.60);
      glVertex3f(0.150, -0.150, -19.0);// Right And Down One Unit (Bottom Right)
      glVertex3f(-0.150, -0.150, -19.0);// Left And Down One Unit (Bottom Left)
      glEnd();
      glEnable(GL_COLOR_MATERIAL);
      glEnable(GL_LIGHTING);
      glPopMatrix();
      side = qMax(w_width, w_height);
      glViewport((w_width - side) / 2, (w_height - side) / 2, side, side);
   };
}


void GLWidget::lastdatapos_timer_slot()
{
   QString StatusMessage;
   QString out_image;
   long current_dataset;
   current_dataset = NewGLContainer.update_last_position();
   
   if (lastpos_record != current_dataset)
   {
   emit currentFrame(current_dataset);
   
   
   lastpos_record = current_dataset;
   updateGL();
   }
};


void GLWidget::slideshow_timer_slot()
{
  if(delta_object_time<=0)
    NewGLContainer.get_delta_time(&delta_object_time);

  int dTreal=elapsed_system_time.elapsed(); // Time left since start of "Play Frames"
  double speed=slideshow_time; // Realtime speed of "Play Frames"
  double dt=delta_object_time; // Time offset of datasets
  int dr=(long)(speed*dTreal/(1000*dt)); // Delta row of dataset since start of "Play Frames"
  int r=NewGLContainer.beginning_record+dr; // Row to draw

  static float lastspeed;
  if(lastspeed!=speed) {
    lastspeed=speed;
    slideshow_timer->stop();  
    elapsed_system_time.start();
    slideshow_timer->start(1);
    NewGLContainer.beginning_record=r;
    return;
  }

  int maxr=NewGLContainer.get_max_records()-2;
  if(r>maxr) {
    slideshow_timer->stop();  
    elapsed_system_time.start();
    slideshow_timer->start(1);
    NewGLContainer.beginning_record=0;
    r=0;
  }
    
  if(lastr!=r) {
    lastr=r;
    NewGLContainer.set_update_position(r);
    emit currentFrame(r);
    dataset_slider->setRange(0, maxr);
//    updateGL();  
  }
};

void GLWidget::button_lastpos_slot()
{

   if (lastdatapos_timer_running == false )
   {
      lastdatapos_timer_running = true;
      slideshow_timer_running = false;
      lastdatapos_timer->stop();
      slideshow_timer->stop();
      lastpos_record =0;
      lastdatapos_timer->start(15);
      StatusBar->showMessage("Timer started");
      emit animbuttondown(false);
      emit startvisbuttondown(true);
   }
   else
   {
      lastdatapos_timer_running = false;
      slideshow_timer_running = false;
      slideshow_timer->stop();
      lastdatapos_timer->stop();
      StatusBar->showMessage("Timer stopped");
      emit animbuttondown(false);
      emit startvisbuttondown(false);
   };

};

/*
void GLWidget::button_slideshow_beginning_slot()
{
   QString info;
   if (slideshow_timer_running == false)
   {
     slideshow_timer_running = true;
     lastdatapos_timer->stop();
     slideshow_timer->stop();  
     NewGLContainer.get_delta_time(&delta_object_time);
     NewGLContainer.total_beginning_flag = true;
     elapsed_system_time.start();     
     slideshow_timer->start(50);  
     StatusBar->showMessage("Timer started");
   }
   else
   {
      slideshow_timer_running = false;
      slideshow_timer->stop();
      lastdatapos_timer->stop();
      StatusBar->showMessage("Timer stopped");
   };
};
*/

void GLWidget::button_slideshow_current_slot()
{

 long current_dataset;

   if (slideshow_timer_running == false )
   {
     slideshow_timer_running = true;
     lastdatapos_timer_running = false;
     lastdatapos_timer->stop();
     slideshow_timer->stop();  
     NewGLContainer.get_delta_time(&delta_object_time);
     current_dataset = NewGLContainer.get_current_record();  
     if(current_dataset>=NewGLContainer.update_last_position()) current_dataset=0;
     NewGLContainer.total_beginning_flag = false;
     NewGLContainer.beginning_record = current_dataset;
     elapsed_system_time.start();     
     slideshow_timer->start(1);
     StatusBar->showMessage("Timer started");
     emit animbuttondown(true);
     emit startvisbuttondown(false);
   }
   else
   {
      slideshow_timer_running = false;
      lastdatapos_timer_running = false;
      slideshow_timer->stop();
      lastdatapos_timer->stop();
      StatusBar->showMessage("Timer stopped");
      emit animbuttondown(false);
      emit startvisbuttondown(false);
   };

};

void GLWidget::dataset_update_min_max()
{

   long MaxRecords;
   QString MaxRangeString;
   MaxRecords = NewGLContainer.get_max_records();
   int value;
   if (MaxRecords >2)
   {
      MaxRangeString.setNum(MaxRecords-2);
      MaxRangeString = "Slider updated from 0 to " +  MaxRangeString ;
      StatusBar->showMessage(MaxRangeString);
      dataset_slider->setRange ( 0, MaxRecords-2)    ;
      value = dataset_slider->sliderPosition();
      NewGLContainer.set_update_position(value);
      updateGL();
   }
   else
   {
      StatusBar->showMessage("Not yet any valid position files");
      dataset_slider->setSliderPosition(0);
   };

};

void GLWidget:: dataset_update_position( int value   )
{
   long MaxRecords;
//   QString CurrentSelectedFrame;
   MaxRecords = NewGLContainer.get_max_records();
   if (MaxRecords >2)
   {
//      CurrentSelectedFrame.setNum(value);
//      CurrentSelectedFrame = "Selected frame: " + CurrentSelectedFrame;
//      StatusBar->showMessage(CurrentSelectedFrame);
      NewGLContainer.set_update_position(value);
//      updateGL();
   }
}

void GLWidget:: dataset_update_position_spinbox( int value   )
{

   long MaxRecords;
//   QString CurrentSelectedFrame;
   MaxRecords = NewGLContainer.get_max_records();
   if (MaxRecords >2)
   {
      dataset_spinbox->setRange ( 0, MaxRecords-2)    ;
//      CurrentSelectedFrame.setNum(value);
//      CurrentSelectedFrame = "Selected frame: " + CurrentSelectedFrame;
//      StatusBar->showMessage(CurrentSelectedFrame);
      NewGLContainer.set_update_position(value);
      updateGL();
   }
   else
   {
      dataset_spinbox->setValue ( 0 );
   };
}

void GLWidget:: set_slideshow_time( int value   )
{
  slideshow_time=pow(10,value/(150/3.0));
  TimeSlider->setSingleStep(slideshow_time/10);
}

void GLWidget:: set_slideshow_time( double value   )
{
  slideshow_time=value;
  TimeSlider->setSingleStep(slideshow_time/10);
}


void GLWidget::show_object_info()
{
   static QString Object_infos;
   NewGLContainer.get_object_info(Object_infos);
   direct_output->clear();
   direct_output->insertPlainText(Object_infos);
   direct_output->textCursor().movePosition(QTextCursor::End);
}


void GLWidget::listboxclicked()
{
   QString InfoListItem;
   InfoListItem.setNum(ListBox-> currentRow());
   NewGLContainer.set_current_object(ListBox-> currentRow());
   update_object_properties();
   show_object_info();
   follow_object = false;
   updateGL();
}

void GLWidget::update_object_properties()
{
   bool object_enabled, object_selected, object_contour, object_path, object_local_cos;
   int mode;
   NewGLContainer.get_object_properties(&object_enabled, &object_selected, &object_contour, &object_path, &object_local_cos, &mode);
   if (object_enabled == true) EnabledCheckBox->setChecked(true); else EnabledCheckBox->setChecked(false);
   if (object_selected == true) SelectedCheckBox->setChecked(true); else SelectedCheckBox->setChecked(false);
   if (object_contour == true) ContourCheckBox->setChecked(true); else ContourCheckBox->setChecked(false);
   if (object_path == true) PathCheckBox->setChecked(true); else PathCheckBox->setChecked(false);
   if (object_local_cos == true) LocalCosCheckBox->setChecked(true); else LocalCosCheckBox->setChecked(false);
   switch(mode) {
     case 0: Filled_Radio_Button->setChecked(true); break;
     case 1: Line_Radio_Button->setChecked(true); break;
     case 2: Points_Radio_Button->setChecked(true); break;
   }
}

void GLWidget::properties_changed()
{

   bool object_enabled, object_selected, object_contour, object_path, object_local_cos;
   //if (EnabledCheckBox->isChecked() == true)  object_enabled =true; else object_enabled =false;
   //if (SelectedCheckBox->isChecked() == true)  object_selected =true; else object_selected =false;
   //if (ContourCheckBox->isChecked() == true)  object_contour =true; else object_contour =false;
   //if (LocalCosCheckBox->isChecked() == true)  object_local_cos =true; else object_local_cos =false;
   //NewGLContainer.set_object_properties(object_enabled, object_selected, object_contour, object_local_cos);
   int counter= 0;
   int number_of_objects = 0;

         if (EnabledCheckBox->isChecked() == true)  object_enabled =true; else object_enabled =false;
         if (SelectedCheckBox->isChecked() == true)  object_selected =true; else object_selected =false;
         if (ContourCheckBox->isChecked() == true)  object_contour =true; else object_contour =false;
         if (PathCheckBox->isChecked() == true)  object_path =true; else object_path =false;
         if (LocalCosCheckBox->isChecked() == true)  object_local_cos =true; else object_local_cos =false;
         
           

           number_of_objects =  ListBox-> count();  
         while (counter < number_of_objects){  // here, ALL object properties are changed FW
         if (ListBox->isItemSelected(ListBox->item(counter))){  
                 NewGLContainer.set_current_object(counter);
                 NewGLContainer.set_object_properties(object_enabled, object_selected, object_contour, object_path, object_local_cos);
         }
           counter ++;
         }
   
   updateGL();
}

void GLWidget::writePixmaps()
{
  float dt; // Simulation output time (t2-t1)
  NewGLContainer.get_delta_time(&dt);
  float s; // Anim.-Speed
  s=slideshow_time;
  int f; // Video frames per second
  bool ok;
  f=QInputDialog::getInteger(this, tr("Write Pixmaps"), tr("Write a sequence of PNG-images\nfor future processing to a video.\n\n(Do not move the window out of the desktop!)\n\nUsing the current anim.-speed and\n? frames per second for output:"), 25, 1, 2147483647, 1, &ok);
  if(ok==false) return;
  if(s/dt/f<1) {
    int ret=QMessageBox::warning(this, tr("Write Pixmaps"), tr("Some video-frames would contain the same data,\nbecause the anim.-speed is to slow,\nand/or the framerate is to high\nand/or the time-intervall of the pos-files is to high.\n\nContinue anyway?"), QMessageBox::Yes, QMessageBox::No);
    if(ret==QMessageBox::No) return;
  }
  // delete pixmap_*.png
  QDir pixmapfiles(".","pixmap_*.png");
  for(unsigned int i=0;i<pixmapfiles.count();++i)
    pixmapfiles.remove(pixmapfiles[i]);

  int i=0; // Video Frame
  int j; // Simulation Frame
  int last_sim_frame = NewGLContainer.update_last_position();
  int last_video_frame = int(dt*f/s*last_sim_frame);
  for(j=0; j<=last_sim_frame; j=int(s/dt/f*++i)) {
    printf("Output Video-Frame %06d/%06d = Anim.-Frame %06d/%06d\n", i, last_video_frame, j, last_sim_frame);
    NewGLContainer.set_update_position(j);
    updateGL();
    write_output_image(QString("pixmap_%1.png").arg(i,6,10,QLatin1Char('0')));
  }

}

void GLWidget::follow_object_slot()
{

   float invert[16];
   float minimum_x;
   float maximum_x;
   float minimum_y;
   float maximum_y;
   float minimum_z;
   float maximum_z;
   float positions[3];
   float rotations[3];
   double scale_factor;

   glMatrixMode( GL_MODELVIEW);
   glPushMatrix();
   glLoadIdentity();
   NewGLContainer.get_current_trans(positions);
   NewGLContainer.get_current_rot(rotations);
   NewGLContainer.get_object_dimensions(& minimum_x,& maximum_x, & minimum_y, & maximum_y,& minimum_z, & maximum_z);
   glTranslatef(positions[0],positions[1], positions[2]);
   KardanMatrixInvert(rotations,invert);
   glMultMatrixf(invert);
   scale_factor =  0.8 *0.150 / (maximum_x);
   glScaled(scale_factor, scale_factor, scale_factor);
   glTranslatef(-positions[0],-positions[1], -positions[2]);
   glGetFloatv( GL_MODELVIEW_MATRIX, followState );
   glPopMatrix();
   glPushMatrix();
   glLoadIdentity();
   glGetDoublev( GL_MODELVIEW_MATRIX, saveState );
   glPopMatrix();
   follow_object = true;
   updateGL();
}

void GLWidget::set_mode_filled_slot()
{
   NewGLContainer.set_draw_mode(0);
   updateGL();
}

void GLWidget::set_mode_line_slot()
{
   NewGLContainer.set_draw_mode(1);
   updateGL();
}
void GLWidget::set_mode_point_slot()
{
   NewGLContainer.set_draw_mode(2);
   updateGL();
}

void GLWidget::write_current_image() {
  write_output_image("pixmap.png");
}

void GLWidget::write_output_image(QString out_filename)
{
// pixmap = renderPixmap(width(), height(), false);
// pixmap.save(out_filename, "PNG", 0);

  QImage image = grabFrameBuffer();
  image.save(out_filename, "PNG", 0);
}

void GLWidget::setAnimSpeed(int value) {
  emit newAnimSpeed(pow(10,value/(150/3.0)));
}

void GLWidget::setSlider(double value) {
  emit newSliderValue(int(150/3.0*log10(value)));
}
