/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <GL/gl.h>
#include "ketsim.h" // class's header file
#include <QImage>
#include <QDialog>
#include <QDoubleSpinBox>
#include <QLayout>
#include <QPushButton>
#include <QLabel>
#include <mainwindow.h>

using namespace std;

#ifndef CALLBACK
#define CALLBACK
#endif

// Windows: void CALLBACK beginCallback(GLenum which)
void CALLBACK beginCallbackketsim(GLenum which)
{
   glBegin(which);
}
void CALLBACK errorCallbackketsim(GLenum errorCode)
{
  printf("Tesselation-Error in 'ketsim.cpp'. Error-Code: %d\nError-String: %s\n",errorCode,gluErrorString(errorCode));
}

void CALLBACK endCallbackketsim(void)
{
   glEnd();
}

void CALLBACK vertexCallbackketsim(GLdouble *vertex)
{
   glVertex3dv(vertex);
}

void CALLBACK combineCallbackketsim(GLdouble coords[3],
				     GLdouble *vertex_data[4],
				     GLfloat weight[4], GLdouble **dataOut )
{
   GLdouble *vertex;
   vertex = (GLdouble *) malloc(7 * sizeof(GLdouble));
   vertex[0] = coords[0];
   vertex[1] = coords[1];
   vertex[2] = coords[2];
   vertex[3] = 0.0;
   vertex[4] = 0.0;
   vertex[5] = 0.0;
   vertex[6] = 0.0;
   *dataOut = vertex;
   printf("Intersecting polygon in 'ketsim.cpp'. Should not happen. Check contours.\n");
   void *dummy; dummy=vertex_data; dummy=weight; // to prevent warning of unused parameter
}

/*
// Dose not work with Mesa
void CALLBACK edgeFlagDataketsim(
				  GLboolean flag,
				  void* polygon_data)
{
   if (flag == true) glEdgeFlag(GL_TRUE); else glEdgeFlag(GL_FALSE);
   void *dummy;dummy=polygon_data; // to avoid warning: unusend paramerter
};
*/

bool KetSim::initTextures=false;
GLubyte *KetSim::amlogo[10];
int KetSim::amlogoWH[10];

// class constructor

KetSim::KetSim(char* body_name, char * pos_file) : CBody( body_name,  pos_file), Contour_List(0)
{
   std::ifstream ketsimcosys("KETSIM_CoSys.dat");
   if(ketsimcosys.is_open()) {
     float a, b, g;
     ketsimcosys>>initTransX>>initTransY>>initTransZ;
     ketsimcosys>>a>>b>>g;
     ketsimcosys>>initScale;
     ketsimcosys.close();
     initRotMatrix[ 0]=cos(b)*cos(g);
     initRotMatrix[ 1]=cos(a)*sin(g)+sin(a)*sin(b)*cos(g);
     initRotMatrix[ 2]=sin(a)*sin(g)-cos(a)*sin(b)*cos(g);
     initRotMatrix[ 3]=0.0;
     initRotMatrix[ 4]=-cos(b)*sin(g);
     initRotMatrix[ 5]=cos(a)*cos(g)-sin(a)*sin(b)*sin(g);
     initRotMatrix[ 6]=sin(a)*cos(g)+cos(a)*sin(b)*sin(g);
     initRotMatrix[ 7]=0.0;
     initRotMatrix[ 8]=sin(b);
     initRotMatrix[ 9]=-sin(a)*cos(b);
     initRotMatrix[10]=cos(a)*cos(b);
     initRotMatrix[11]=0.0;
     initRotMatrix[12]=0.0;
     initRotMatrix[13]=0.0;
     initRotMatrix[14]=0.0;
     initRotMatrix[15]=1.0;
   } else {
     initTransX=initTransY=initTransZ=0;
     initScale=1;
     memset(initRotMatrix, 0, sizeof(float)*4*4);
     initRotMatrix[0]=1;
     initRotMatrix[5]=1;
     initRotMatrix[10]=1;
     initRotMatrix[15]=1;
   }

   position_x  = 0.0;
   position_y  = 0.0;
   position_z  = 0.0;
   angle_alpha = 0.0;
   angle_beta  = 0.0;
   angle_gamma = 0.0;
   color[0]     =  1.0;
   color[1]     =  0.0;
   color[2]     =  0.0;
   color[3]     =  1.0;
   colorvalue = 0.0;
   color_shininess = 20.0;
   draw_color=true;
   scale_factor=1.0;
   current_time = 0.0;
   x_min = 100000000000.0;
   x_max = -100000000000.0;
   y_min = 100000000000.0;
   y_max = -100000000000.0;
   z_min = 100000000000.0;
   z_max = -100000000000.0;
   posfile_ready = false;
   recordsize = 0;
   posfile_pointer = NULL;
   Contour_List = 0;
   Displ_List = 0;
   force=0;

   if(initTextures==false) {
     initTextures=true;
     for(int k=0; k<10; ++k) {
       QIcon icon(":/amlogo.svg");
       amlogoWH[k]=512/(1<<k);
       amlogo[k]=new GLubyte[amlogoWH[k]*amlogoWH[k]*4];
       QImage img(icon.pixmap(amlogoWH[k]).toImage());
       int i, j;
       for(i=0; i<amlogoWH[k]; i++)
         for(j=0; j<amlogoWH[k]; j++) {
           QRgb rgba=img.pixel(i,j);
           amlogo[k][i*amlogoWH[k]*4+j*4+0]=qRed(rgba);
           amlogo[k][i*amlogoWH[k]*4+j*4+1]=qGreen(rgba);
           amlogo[k][i*amlogoWH[k]*4+j*4+2]=qBlue(rgba);
           amlogo[k][i*amlogoWH[k]*4+j*4+3]=qAlpha(rgba);
         }
     }
   }
} 

// class destructor
KetSim::~KetSim()
{
   if (Displ_List) glDeleteLists(Displ_List, 1);
   if (Contour_List) glDeleteLists(Contour_List, 1);
}

void KetSim::draw()
{
   glLineWidth(1.0);
   glPointSize(2.0);
   glEnable(GL_LINE_SMOOTH);
   glEnable(GL_POINT_SMOOTH);

   if (mode ==0) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   if (mode ==1) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   if (mode ==2) glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);

   glDisable(GL_COLOR_MATERIAL);
   glEnable(GL_LIGHTING);
   updateRotationMatrix();
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();

   //initial translation and roation
   glTranslatef(initTransX, initTransY, initTransZ);
   glMultMatrixf(initRotMatrix);
   glScalef(initScale, initScale, initScale);


   glPolygonOffset(OFFSETFAC0,OFFSETUNIT1);
   glEnable(GL_POLYGON_OFFSET_FILL);
   glEnable(GL_POLYGON_OFFSET_LINE);
   glEnable(GL_POLYGON_OFFSET_POINT);

   glEnable(GL_BLEND);
   glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
   translateColorvalueToRGBA(colorvalue, &color[0], &color[1], &color[2], &color[3]);
   glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
   GLfloat mat_specular[] = {1.0,1.0,1.0, 1.0};
   glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
   float em[4]={0,0,0,0};
   glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);
   glTranslated(position_x, position_y, position_z);
   glMultMatrixf(object_rotation_matrix);
   glCallList(Displ_List);
   glDisable(GL_POLYGON_OFFSET_FILL);
   glPolygonOffset(0.0,0.0);
   if (contour_flag==true)
   {
      draw_contour();
   };

   if (selected_flag==true)
   {
      glPolygonOffset(OFFSETFAC1,OFFSETUNIT1);
      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glColor3f(0.0, 1.0, 0.0);
      glLineWidth(4.0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_min, y_max, z_min);
      glVertex3f(x_min, y_max, z_max);
      glVertex3f(x_min, y_min, z_max);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_max, y_min, z_min);
      glVertex3f(x_max, y_max, z_min);
      glVertex3f(x_max, y_max, z_max);
      glVertex3f(x_max, y_min, z_max);
      glVertex3f(x_max, y_min, z_min);
      glEnd();
      glBegin(GL_LINES);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_max, y_min, z_min);
      glVertex3f(x_min, y_max, z_min);
      glVertex3f(x_max, y_max, z_min);
      glVertex3f(x_min, y_max, z_max);
      glVertex3f(x_max, y_max, z_max);
      glVertex3f(x_min, y_min, z_max);
      glVertex3f(x_max, y_min, z_max);
      glEnd();
      glEnable(GL_LIGHTING);
      glPolygonOffset(0.0,0.0);
      glDisable(GL_COLOR_MATERIAL);
   };


   if (local_cos_flag==true)
   {
      float abs=fmax(fmax(x_max,y_max),z_max);

      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glLineWidth(3.0);
      // X DIR red
      glColor3ub(255, 0, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(abs* 1.5, 0.0, 0.0);
      glEnd();
      // Y DIR green
      glColor3ub(0, 255, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, abs* 1.5, 0.0);
      glEnd();
      // Z DOR blue
      glColor3ub(0, 0, 255);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, 0.0, abs* 1.5);
      glEnd();
      glEnable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
   };


   glDisable(GL_POLYGON_OFFSET_FILL);
   glDisable(GL_POLYGON_OFFSET_LINE);
   glDisable(GL_POLYGON_OFFSET_POINT);
   glPolygonOffset(0.0,0.0);
   glPopMatrix();


  if(path_flag) {
    long drawpathto=curdataset;
    if(path.size()<=(unsigned long)drawpathto) {
      for(int i=path.size(); i<=drawpathto; i++) {
        update(i);
        Point3D *p=new Point3D;
        p->x=position_x;
        p->y=position_y;
        p->z=position_z;
        path.push_back(p);
      }
    }
    drawPath(0, 1, 0);
  }
}

void KetSim::updateminmax(float vector[3])
{
   if   (vector[0]<x_min) x_min = vector[0];
   if   (vector[0]>x_max) x_max = vector[0];
   if   (vector[1]<y_min) y_min = vector[1];
   if   (vector[1]>y_max) y_max = vector[1];
   if   (vector[2]<z_min) z_min = vector[2];
   if   (vector[2]>z_max) z_max = vector[2];
};


void KetSim::updateminmax(double vector[3])
{
   if   (vector[0]<x_min) x_min = vector[0];
   if   (vector[0]>x_max) x_max = vector[0];
   if   (vector[1]<y_min) y_min = vector[1];
   if   (vector[1]>y_max) y_max = vector[1];
   if   (vector[2]<z_min) z_min = vector[2];
   if   (vector[2]>z_max) z_max = vector[2];
}

void KetSim::updateRotationMatrix()
{
   object_rotation_matrix[0]  = cos (angle_beta)  * cos (angle_gamma) ;
   object_rotation_matrix[1]  = cos (angle_alpha) * sin (angle_gamma) + sin (angle_alpha) * sin (angle_beta) * cos(angle_gamma);  ;
   object_rotation_matrix[2]  = sin (angle_alpha) * sin (angle_gamma) - cos (angle_alpha) * sin (angle_beta)*cos(angle_gamma) ;
   object_rotation_matrix[3]  = 0.0 ;
   object_rotation_matrix[4]  =-cos (angle_beta)  * sin (angle_gamma) ;
   object_rotation_matrix[5]  = cos (angle_alpha) * cos (angle_gamma) - sin (angle_alpha) * sin (angle_beta) * sin (angle_gamma) ;
   object_rotation_matrix[6]  = sin (angle_alpha) * cos (angle_gamma) + cos(angle_alpha) * sin (angle_beta) * sin (angle_gamma) ;
   object_rotation_matrix[7]  = 0.0 ;
   object_rotation_matrix[8]  = sin (angle_beta) ;
   object_rotation_matrix[9]  =-sin (angle_alpha) * cos (angle_beta);
   object_rotation_matrix[10] = cos (angle_alpha) * cos (angle_beta);
   object_rotation_matrix[11] = 0.0 ;
   object_rotation_matrix[12] = 0.0 ;
   object_rotation_matrix[13] = 0.0 ;
   object_rotation_matrix[14] = 0.0 ;
   object_rotation_matrix[15] = 1.0 ;
}

void KetSim::read_data()
{
}

void KetSim::update(long dataset_number)
{
   int erg;
   if (posfile_ready==false)getFilePointerandRecSize();
   if (posfile_ready==true)
   {

      fseek (posfile_pointer, dataset_number * recordsize ,SEEK_SET);
      if (pos_file_binary==false)
      {

	 char szLineBuffer[charlength];
	 erg=GetCharacterLine(szLineBuffer, charlength, posfile_pointer);
	 if (erg>0)
	 {
	    sscanf(szLineBuffer, "%f %f %f %f %f %f\n",
		   &current_time,
		   &position_x,
		   &position_y,
		   &position_z,
		   &angle_gamma,
		   &force);
	 };
      }
      else //binary file
      {
	 float tempfloat[6];
	 fread (tempfloat,sizeof(float),6, posfile_pointer);
	 current_time   =tempfloat[0];
	 position_x      =tempfloat[1];
	 position_y  =tempfloat[2];
	 position_z  =tempfloat[3];
	 angle_gamma  =tempfloat[4];
	 force =tempfloat[5];
      };//binary file
      angle_alpha  = 0.0;
      angle_beta   = 0.0;
      colorvalue=force2color(force);
   };
   curdataset=dataset_number;
}

long KetSim::get_dataset()
{
   long recordsizereturn;
   if (posfile_ready==false)getFilePointerandRecSize();
   if (posfile_ready==true)
   {
      recordsizereturn = (ftell(posfile_pointer)/recordsize) ;
      return recordsizereturn;
   }else
   {
      return 0;
   };

}

long KetSim::get_dataset_last()
{

   long recordsizereturn;
   if (posfile_ready==false)getFilePointerandRecSize();
   if (posfile_ready==true)
   {
      recordsizereturn = (getlastposition(posfile_pointer)/recordsize) ;
      return recordsizereturn;
   }else
   {
      return 0;
   };
}


long KetSim::getrecordsizetext(FILE * textfilepointer)
{
   long oldposition, pos_before_cr, newposition;
   int ch;
   oldposition=ftell (textfilepointer);
   fseek (textfilepointer,0,SEEK_SET);
   while (true)
   {
      ch = fgetc(textfilepointer);
      if (ch != EOF)
      {
	 if ((ch == CR) || (ch == LF))
	 {
	    pos_before_cr=ftell (textfilepointer);
	    ch = fgetc(textfilepointer);
	    //                             printf("Found CR / LF before %ld\n",ftell (textfilepointer));
	    if ((ch == CR) || (ch == LF)) {}else
	    {
	       fseek (textfilepointer,pos_before_cr,SEEK_SET);
	    };
	    newposition=ftell (textfilepointer);
	    break;
	 }; // Get out - return success
      }
      else
      {
	 fseek (textfilepointer,oldposition,SEEK_SET);
	 return 0;
	 break;
      };
   };
   fseek (textfilepointer,oldposition,SEEK_SET);
   return (newposition);
};

long KetSim::getlastposition(FILE * filestream)
{
   long curpos;
   long lastpos;
   curpos=ftell(filestream);
   fseek (filestream,0,SEEK_END);
   lastpos =ftell(filestream);
   fseek (filestream, curpos, SEEK_SET);
   return lastpos;
};

bool KetSim::getFilePointerandRecSize()
{
   posfile_ready=true;
   FILE * tempfilepointer;
   if(posfile_pointer==0)
     tempfilepointer = fopen (posfile_name,"rb");
   else
     tempfilepointer = posfile_pointer;
   if (tempfilepointer>0)
   {
      if (pos_file_binary==false)
      {
	 long temprecordsize = getrecordsizetext(tempfilepointer);
	 if (temprecordsize>0)
	 {
	    recordsize=temprecordsize;
	    posfile_pointer=tempfilepointer;
	 }
	 else
	 {
	    posfile_ready=false;
	    fclose (tempfilepointer);
	 }; //recordsize ==0
      }else //binary
      {
	 if (getlastposition(tempfilepointer)/(sizeof(float)*6)>0)
	 {
	    recordsize=sizeof(float)*6;
	    posfile_pointer=tempfilepointer;
	 }
	 else
	 {
	    posfile_ready=false;
	    fclose (tempfilepointer);
	 }; //no data sets in file

      };
   } else
   {
      posfile_ready=false;
   };//tempfilepointer<=0

   return  posfile_ready;
}

void KetSim::init()
{
   tobj = gluNewTess();
#ifdef UNIX
   gluTessCallback(tobj, GLU_TESS_VERTEX, (GLvoid (*)())  vertexCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_BEGIN, (GLvoid (*)())  beginCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_END,  (GLvoid (*)())  endCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_ERROR,  (GLvoid (*)()) errorCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_COMBINE,  (GLvoid (*)()) combineCallbackketsim);
//   gluTessCallback(tobj, GLU_TESS_EDGE_FLAG_DATA,  (GLvoid (*)())  edgeFlagDataketsim);
#elif defined WINDOWS
   gluTessCallback(tobj, GLU_TESS_VERTEX, (void (__stdcall *)(void))    vertexCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_BEGIN, (void (__stdcall *)(void))    beginCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_END,  (void (__stdcall *)(void))   endCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_ERROR,  (void (__stdcall *)(void))   errorCallbackketsim);
   gluTessCallback(tobj, GLU_TESS_COMBINE,  (void (__stdcall *)(void))   combineCallbackketsim);
//   gluTessCallback(tobj, GLU_TESS_EDGE_FLAG_DATA, (void (__stdcall *)(void))   edgeFlagDataketsim);
#endif
glFrontFace( GL_CCW);
}

void KetSim::ret_time(float* time)
{
   *time=current_time;
}

void KetSim::ret_trans(float* translation)
{
   translation[0]=position_x  ;
   translation[1]=position_y ;
   translation[2]=position_z  ;
}

void KetSim::ret_rot(float* rotation)
{
   rotation[0]=angle_alpha;
   rotation[1]=angle_beta;
   rotation[2]=angle_gamma;
}

void KetSim::return_max_dimensions(float* min_x, float* max_x, float* min_y, float* max_y,float* min_z, float* max_z)
{
   *min_x = x_min;
   *max_x = x_max;
   *min_y = y_min;
   *max_y = y_max;
   *min_z = z_min;
   *max_z = z_max;
}

void KetSim::draw_pure()
{
   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   glDisable(GL_LIGHTING);
   glEnable(GL_COLOR_MATERIAL);
   glColor3fv(pick_color_value);
   glPushMatrix();
   glTranslated(position_x, position_y, position_z);
   glMultMatrixf(object_rotation_matrix);
   glScaled(scale_factor, scale_factor, scale_factor);
   glCallList(Displ_List);
   glPopMatrix();
   glEnable(GL_LIGHTING);
   glDisable(GL_COLOR_MATERIAL);
}

void KetSim::return_info_string(QString &info)
{
   char degchar[2]={0xC2, 0xB0};
   QString deg=QString::fromUtf8(degchar,2);
   float angledeg=angle_gamma*180/M_PI;
   info=tr("Class: %1\nBody: %2\nPos.: %3\nx: %4mm\ny: %5mm\nz: %6mm\nAngle: %7%8 (%9%10)").arg(class_name).arg(body_file_name).arg(posfile_name).arg(position_x*1000).arg(position_y*1000).arg(position_z*1000).arg(angledeg).arg(deg).arg(angledeg-int(angledeg/360)*360).arg(deg);
}


float KetSim::force2color(float force)
{
  float color=(force-forcemin)/(forcemax-forcemin);
  if(color>1) color=1;
  if(color<0) color=0;
  return color;
}

/*
void KetSim::getMinMax(QString text, float *min, float *max) {
  QDialog dia;
  dia.setWindowTitle(text);
  QGridLayout layout;
  dia.setLayout(&layout);
  QLabel title(tr("Set force-range for objects colors of type:\n%1").arg(text)), minLab(tr("Min.-Value:")), maxLab(tr("Max.-Value:"));
  layout.addWidget(&title,0,0,1,2);
  layout.addWidget(&minLab,1,0,1,1);
  layout.addWidget(&maxLab,2,0,1,1);
  QDoubleSpinBox minBox, maxBox;
  minBox.setRange(-1e50,1e50);
  maxBox.setRange(-1e50,1e50);
  minBox.setValue(*min);
  maxBox.setValue(*max);
  layout.addWidget(&minBox,1,1,1,1);
  layout.addWidget(&maxBox,2,1,1,1);
  QPushButton ok(tr("OK")), reject(tr("Abort"));
  layout.addWidget(&ok,3,0,1,1);
  layout.addWidget(&reject,3,1,1,1);
  connect(&ok, SIGNAL(clicked()), &dia, SLOT(accept()));
  connect(&reject, SIGNAL(clicked()), &dia, SLOT(reject()));
  if(dia.exec()==QDialog::Accepted) {
    *min=minBox.value();
    *max=maxBox.value();
  }
}*/

vector<KetSim::vertex_data_struct*> KetSim::recalcContour(vector<KetSim::vertex_data_struct*> &vertex_data) {
  vector<Contour*> contour;
  vector<Contour*> newcont;
  vector<vertex_data_struct*> newvertex_data;
  float Z=0;

  // Copy Contour
  Contour *c;
  vector<vertex_data_struct*>::iterator i;
  for(i=vertex_data.begin(); i!=vertex_data.end(); i++) {
    c=new Contour;
    c->x=(*i)->vertex_points[0];
    c->y=(*i)->vertex_points[1];
    Z=(*i)->vertex_points[2];
    c->b=(*i)->border_flag;
    contour.push_back(c);
  }

  // recalc Contour
  newcont=CBody::recalcContour(contour);

  // Copy Contour
  vertex_data_struct *d;
  vector<Contour*>::iterator j;
  for(j=newcont.begin(); j!=newcont.end(); j++) {
    d=new vertex_data_struct;
    d->vertex_points=new float[3];
    d->vertex_points[0]=(*j)->x;
    d->vertex_points[1]=(*j)->y;
    d->vertex_points[2]=Z;
    d->border_flag=(*j)->b;
    newvertex_data.push_back(d);
  }
  
  // Delete
  for(i=vertex_data.begin(); i!=vertex_data.end(); i++) {
    delete[](*i)->vertex_points;
    delete (*i);
  }
  for(j=contour.begin(); j!=contour.end(); j++)
    delete (*j);
  
  return newvertex_data;
}
