/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "mainwindow.h"
#include <QDoubleSpinBox>
#include <QDockWidget>
#include <sstream>

using namespace std;

MainWindow::MainWindow(bool minDocks, QString wst, QString viewFile, bool autoPlay, double animationSpeed)
{
   setWindowIcon(QIcon(":/amlogo.svg"));
   helpDialog=NULL;
   centralWidget = new QWidget;
   setCentralWidget(centralWidget);
   ObjectPropertyBox = new QDockWidget(tr("Body Properties"),this);
   ObjectPropertyBox->setObjectName("Body_Properties");
   output_edit = new QTextEdit;
   output_edit->setLineWrapMode(QTextEdit::NoWrap);
   output_edit->setMinimumSize(0, 60);
   LeftInfoListDock=new QDockWidget(tr("Bodies"),this);
   LeftInfoListDock->setObjectName("Bodies");
   LeftInfoList = new QListWidget(LeftInfoListDock);
   QGridLayout *LeftInfoListLayout=new QGridLayout;
   QWidget *LeftInfoListContent=new QWidget;
   LeftInfoListDock->setWidget(LeftInfoListContent);
   LeftInfoListContent->setLayout(LeftInfoListLayout);
   LeftInfoListLayout->addWidget(LeftInfoList, 0,0);
#if QT_VERSION >= 0x040300
   LeftInfoListLayout-> setContentsMargins(5,5,5,5);
#endif
   LeftInfoList->setSelectionMode(QAbstractItemView::ExtendedSelection);
   LeftInfoList->setMinimumSize(0, 60);
   FrameSpinBox = new QSpinBox;
   FrameSpinBox->setMinimumSize(70, 0);
   FrameSlider = new QSlider;
   timerisrunning=false;
   glWidget = new GLWidget(this);
   glWidget->setMinimumSize(200,200);
   connect(glWidget, SIGNAL(currentFrame(int)), FrameSpinBox, SLOT(setValue(int)));
   glWidget->direct_output=output_edit;
   glWidget->NewGLContainer.direct_output=output_edit;
   glWidget->NewGLContainer.LeftInfoList=LeftInfoList;
   StatusBar = new QStatusBar;
   setStatusBar(StatusBar);
   glWidget->StatusBar  = 	StatusBar;
   QDoubleSpinBox *TimeSliderSpinBox=new QDoubleSpinBox;
   glWidget->TimeSlider=TimeSliderSpinBox;
   TimeSliderSpinBox->setValue(1);
   TimeSliderSpinBox->setRange(0,1e99);
   TimeSliderSpinBox->setSingleStep(0.1);
   TimeSliderSpinBox->setDecimals(7);
   TimeSliderSpinBox->setMinimumSize(70, 0);
   connect(TimeSliderSpinBox, SIGNAL(valueChanged(double)), glWidget, SLOT(set_slideshow_time(double)));
   connect(TimeSliderSpinBox, SIGNAL(valueChanged(double)), glWidget, SLOT(setSlider(double)));
   TimeSlider = new QSlider(Qt::Horizontal);
   connect(glWidget, SIGNAL(newSliderValue(int)), TimeSlider, SLOT(setValue(int)));
   connect(TimeSlider, SIGNAL(sliderMoved(int)), glWidget, SLOT(setAnimSpeed(int)));
   connect(glWidget, SIGNAL(newAnimSpeed(double)), TimeSliderSpinBox, SLOT(setValue(double)));
   TimeFrameLabel = new QLabel;
   TimeFrameLabel->setText("Frame:");
   TimeIntervalLabel = new QLabel;
   TimeIntervalLabel->setText("Anim.-Speed:"); 
   TimeSliderLabel = new QLabel;
   TimeSliderLabel->setText("Interval");
   glWidget->TimeIntervalLabel= TimeIntervalLabel;
   glWidget->TimeSliderLabel= TimeSliderLabel;
   startvis = new QPushButton("L&ast",0);
   startvis->setShortcut(tr("Ctrl+A"));
   startvis->setToolTip(tr("View Last Frame (Ctrl+A)"));
   startvis->setCheckable(true);
   connect(glWidget, SIGNAL(startvisbuttondown(bool)), startvis, SLOT(setChecked(bool)));
   connect(startvis, SIGNAL(clicked()), glWidget, SLOT(button_lastpos_slot()));
   QPushButton *anim_current_button = new QPushButton("&Play");
   anim_current_button->setToolTip(tr("Ctrl+P"));
   anim_current_button->setShortcut(tr("Ctrl+P"));
   anim_current_button->setCheckable(true);
   connect(glWidget, SIGNAL(animbuttondown(bool)), anim_current_button, SLOT(setChecked(bool)));
   connect(anim_current_button, SIGNAL(clicked()), glWidget, SLOT(button_slideshow_current_slot()));
   EnabledCheckBox = new QCheckBox("&Enabled",ObjectPropertyBox );
   EnabledCheckBox->setToolTip(tr("Ctrl+E"));
   EnabledCheckBox->setShortcut(tr("Ctrl+E"));
   SelectedCheckBox = new QCheckBox("Sele&cted",ObjectPropertyBox);
   SelectedCheckBox->setToolTip(tr("Ctrl+C"));
   SelectedCheckBox->setShortcut(tr("Ctrl+C"));
   ContourCheckBox = new QCheckBox("Conto&ur",ObjectPropertyBox);
   ContourCheckBox->setToolTip(tr("Ctrl+U"));
   ContourCheckBox->setShortcut(tr("Ctrl+U"));
   PathCheckBox = new QCheckBox("Pat&h",ObjectPropertyBox);
   PathCheckBox->setToolTip(tr("Ctrl+H"));
   PathCheckBox->setShortcut(tr("Ctrl+H"));
   LocalCosCheckBox = new QCheckBox("&Draw Coordinate System",ObjectPropertyBox);
   LocalCosCheckBox->setToolTip(tr("Ctrl+D"));
   LocalCosCheckBox->setShortcut(tr("Ctrl+D"));
   QRadioButton *Filled_Radio_Button=new QRadioButton(tr("Pol&ygon"));
   Filled_Radio_Button->setToolTip(tr("Ctrl+Y"));
   Filled_Radio_Button->setShortcut(tr("Ctrl+Y"));
   Filled_Radio_Button->setMinimumHeight(15);
   QRadioButton *Line_Radio_Button=new QRadioButton(tr("L&ine"));
   Line_Radio_Button->setToolTip(tr("Ctrl+I"));
   Line_Radio_Button->setShortcut(tr("Ctrl+I"));
   Line_Radio_Button->setMinimumHeight(15);
   QRadioButton *Points_Radio_Button=new QRadioButton(tr("Poi&nt"));
   Points_Radio_Button->setToolTip(tr("Ctrl+N"));
   Points_Radio_Button->setShortcut(tr("Ctrl+N"));
   Points_Radio_Button->setMinimumHeight(15);
   connect(Filled_Radio_Button, SIGNAL(clicked()), glWidget, SLOT(set_mode_filled_slot()));
   connect(Line_Radio_Button, SIGNAL(clicked()), glWidget, SLOT(set_mode_line_slot()));
   connect(Points_Radio_Button, SIGNAL(clicked()), glWidget, SLOT(set_mode_point_slot()));
   QGridLayout *PropertyLayout = new QGridLayout;
#if QT_VERSION >= 0x040300
   PropertyLayout-> setContentsMargins(5,5,5,5);
#endif
   PropertyLayout-> addWidget (output_edit, 0, 0, 1, 2);
   PropertyLayout-> addWidget (EnabledCheckBox, 1, 0 );
   PropertyLayout-> addWidget (SelectedCheckBox, 1, 1 );
   PropertyLayout-> addWidget (ContourCheckBox, 2, 0 );
   PropertyLayout-> addWidget (PathCheckBox, 2, 1 );
   PropertyLayout-> addWidget (LocalCosCheckBox, 3, 0 , 1, 2);
   PropertyLayout-> addWidget (Filled_Radio_Button, 4, 0);
   PropertyLayout-> addWidget (Line_Radio_Button, 4, 1);
   PropertyLayout-> addWidget (Points_Radio_Button, 5, 0);
   QWidget *ObjectPropertyBoxContent=new QWidget;
   ObjectPropertyBoxContent->setLayout(PropertyLayout),
   ObjectPropertyBox->setWidget(ObjectPropertyBoxContent);
   QGridLayout *mainlayout = new QGridLayout ;
#if QT_VERSION >= 0x040300
   mainlayout->setContentsMargins(0,0,5,0);
#endif

   AnimationDock=new QDockWidget(tr("Animation"),this);
   AnimationDock->setObjectName("Animation");
   QWidget *AnimationDockContent=new QWidget;
   AnimationDock->setWidget(AnimationDockContent);
   QGridLayout *AnimationDockLayout=new QGridLayout;
   AnimationDockContent->setLayout(AnimationDockLayout);
#if QT_VERSION >= 0x040300
   AnimationDockLayout->setContentsMargins(5,5,5,5);
#endif
   AnimationDockLayout->addWidget(anim_current_button, 0, 0);
   AnimationDockLayout->addWidget(startvis, 0, 1);
   AnimationDockLayout->addWidget(TimeFrameLabel, 1, 0);
   AnimationDockLayout->addWidget(FrameSpinBox, 1, 1);
   AnimationDockLayout->addWidget(TimeIntervalLabel, 2, 0);
   AnimationDockLayout->addWidget(TimeSliderSpinBox, 2, 1);
   AnimationDockLayout->addWidget(TimeSlider, 3, 0, 1, 2);
   AnimationDockLayout->addItem(new QSpacerItem(0,0,QSizePolicy::Minimum, QSizePolicy::Expanding), 5, 0, 1, 2);

   mainlayout->addWidget(glWidget, 0, 0);
   mainlayout->addWidget(FrameSlider, 0, 1);




   centralWidget->setLayout(mainlayout);
   QDir dir;
   QString title;
   title = "AMVis: " + dir.absolutePath();	
   setWindowTitle(title);

   QDesktopWidget *desktop = new QDesktopWidget;
   QRect DesktopRect = desktop->screenGeometry(desktop->screenNumber(0)) ;
   int DesktopX, DesktopY, DesktopWidth, DesktopHeight;
   DesktopRect.getRect ( &DesktopX, &DesktopY, &DesktopWidth,  &DesktopHeight );
   int actualwith=800;
   int actualheight=600;
   resize(actualwith, actualheight);
   move ((int) ((DesktopWidth/2) - (actualwith/2)),(int) ((DesktopHeight/2) - (actualheight/2) ));


   glWidget->dataset_slider= FrameSlider;
   glWidget->dataset_spinbox = FrameSpinBox;
   connect(FrameSlider, SIGNAL(sliderPressed ()), glWidget, SLOT(dataset_update_min_max()));
   connect(FrameSlider, SIGNAL(sliderMoved ( int  )  ), glWidget, SLOT(dataset_update_position( int  )));
   connect(FrameSlider, SIGNAL(sliderMoved ( int  )  ), FrameSpinBox, SLOT(setValue ( int  )));
   connect(FrameSpinBox, SIGNAL(valueChanged ( int  )  ), FrameSlider, SLOT(setValue ( int  )));
   connect(FrameSpinBox, SIGNAL(valueChanged ( int  )  ), glWidget, SLOT(dataset_update_position_spinbox ( int  )));

   TimeSlider->setRange ( -150, 150)   ;
   TimeSlider->setSliderPosition(0);
   connect(TimeSlider, SIGNAL(sliderMoved ( int  )  ), glWidget, SLOT(set_slideshow_time( int  )));

   glWidget->ListBox=LeftInfoList;
   glWidget->EnabledCheckBox=EnabledCheckBox;
   glWidget->SelectedCheckBox=SelectedCheckBox;
   glWidget->ContourCheckBox=ContourCheckBox;
   glWidget->PathCheckBox=PathCheckBox;
   glWidget->LocalCosCheckBox = LocalCosCheckBox;
   glWidget->Filled_Radio_Button=Filled_Radio_Button;
   glWidget->Line_Radio_Button=Line_Radio_Button;
   glWidget->Points_Radio_Button=Points_Radio_Button;



   connect(EnabledCheckBox, SIGNAL(stateChanged    ( int  )  ), glWidget, SLOT(properties_changed()));
   connect(SelectedCheckBox, SIGNAL(stateChanged    ( int  )  ), glWidget, SLOT(properties_changed()));
   connect(ContourCheckBox, SIGNAL(stateChanged    ( int  )  ), glWidget, SLOT(properties_changed()));
   connect(PathCheckBox, SIGNAL(stateChanged    ( int  )  ), glWidget, SLOT(properties_changed()));
   connect(LocalCosCheckBox, SIGNAL(stateChanged    ( int  )  ), glWidget, SLOT(properties_changed()));
   connect(LeftInfoList, SIGNAL( itemClicked( QListWidgetItem*)), glWidget , SLOT(listboxclicked()));
   connect(LeftInfoList, SIGNAL( itemDoubleClicked( QListWidgetItem*)), &(glWidget->NewGLContainer) , SLOT(editBody()));


   createActions();
   createMenus();


   addDockWidget(Qt::LeftDockWidgetArea,LeftInfoListDock);
   addDockWidget(Qt::LeftDockWidgetArea,ObjectPropertyBox);
   addDockWidget(Qt::LeftDockWidgetArea,AnimationDock);

   QFile stateFile(wst);
   stateFile.open(QIODevice::ReadOnly);
   QByteArray state=stateFile.readAll();
   stateFile.close();
   restoreState(state, 0);

   if(minDocks) {
     LeftInfoListDock->close();
     ObjectPropertyBox->close();
     AnimationDock->close();
   }

   // ToDo: fataler weise ist die Reihenfolge entscheidend, da mehrere auf
   // initializeGL() zugreifen...
   TimeSliderSpinBox->setValue(animationSpeed);
   glWidget->start_auto_play(autoPlay);
   loadViewPortStateFile(viewFile);
}

void MainWindow::help()
{
  if(!helpDialog) {
    helpDialog=new QDialog;
    helpDialog->setWindowIcon(QIcon(":/amlogo.svg"));
    helpDialog->setWindowTitle(tr("AMVis - Help"));
    QGridLayout *layout=new QGridLayout;
    helpDialog->setLayout(layout);
    QPushButton *home=new QPushButton(tr("Home"));
    layout->addWidget(home,0,0);
    QPushButton *backward=new QPushButton(tr("Backward"));
    layout->addWidget(backward,0,1);
    QPushButton *forward=new QPushButton(tr("Forward"));
    layout->addWidget(forward,0,2);
    QTextBrowser *text=new QTextBrowser;
    layout->addWidget(text,1,0,1,3);
    connect(home, SIGNAL(clicked()), text, SLOT(home()));
    connect(forward, SIGNAL(clicked()), text, SLOT(forward()));
    connect(backward, SIGNAL(clicked()), text, SLOT(backward()));
    connect(text, SIGNAL(backwardAvailable(bool)), backward, SLOT(setEnabled(bool)));
    connect(text, SIGNAL(forwardAvailable(bool)), forward, SLOT(setEnabled(bool)));
    text->setSource(QUrl("qrc:/help.xhtml"));
  }
  helpDialog->show();
  helpDialog->raise();
  helpDialog->activateWindow();
  helpDialog->resize(700,500);
}

void MainWindow::about()
{
  QMessageBox mb(this);
  mb.setWindowTitle(tr("AMVis"));
  mb.setText(tr(
    "<h3>AMVis - 3D Multibody Visualisation Program - Version %1.%2.%3</h3>"
    "<p>Copyright &copy; 2006-2008 Institute of Applied Mechanics, Technische Universit&#228;t M&#252;nchen.<br/>"
    "<tt>http://www.amm.mw.tum.de</tt></p>"
    "<p>Licensed under the GPL (see file COPYING or menu Help/Help).</p>"
    "<p>This is free software; see the source for copying conditions.  There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.</p>"
    "<h4>Authors:</h4>"
    "<ul><li>Falko Schuck<li/>"
    "<li>Markus Friedrich <tt>&lt;friedma@amm.mw.tum.de&gt;</tt></li>"
    "<li>Roland Zander</li></ul>"
    "<h4>This program uses:</h4>"
    "<ul>"
    "<li>'Qt' from <tt>http://www.trolltech.com</tt> (License: GPL)</li>"
    "<li>'libwave - Wavefront/GL rendering library' by Dave Pape from <tt>http://www.evl.uic.edu/pape/sw/libwave.html</tt> (Licence: GPL)</li>"
    "<li>'GLE - Tubing and Extrusion Library' by Linas Vepstas from <tt>http://linas.org/gle/</tt> (Licence: GPL, Artistic)</li>"
    "<li>'Qwt - Qt Widgets for Technical Applications' from <tt>http://qwt.sourceforge.net</tt> (Licence: Qwt License (LGPL))</li>"
    "</ul>"
  ).arg(V1).arg(V2).arg(V3));
  QPixmap pm(":/amtumlogo.png");
  mb.setIconPixmap(pm);
  mb.setButtonText(0, tr("OK"));
  mb.exec();
}

void MainWindow::createActions()
{
   exitAct = new QAction(tr("E&xit"), this);
   exitAct->setShortcut(tr("Ctrl+Q"));
   connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

   cachePosFilesAct = new QAction(tr("&Cache pos-Files by OS"), this);
   connect(cachePosFilesAct, SIGNAL(triggered()), this, SLOT(cachePosFiles()));

   exportSinglePNGAct = new QAction(tr("Export &current frame as &PNG"), this);
   connect(exportSinglePNGAct, SIGNAL(triggered()), glWidget, SLOT(write_current_image()));

   exportPNGAct = new QAction(tr("Export &all frames as &PNG sequence..."), this);
   connect(exportPNGAct, SIGNAL(triggered()), glWidget, SLOT(writePixmaps()));

   glWidget->drawWorldCosAct = new QAction(tr("&World Coordinate System"), this);
   glWidget->drawWorldCosAct->setCheckable(true);
   connect(glWidget->drawWorldCosAct, SIGNAL(triggered()), glWidget, SLOT(updateGL()));
   glWidget->drawXYPlaneAct = new QAction(tr("&x-y-Plane"), this);
   glWidget->drawXYPlaneAct->setCheckable(true);
   connect(glWidget->drawXYPlaneAct, SIGNAL(triggered()), glWidget, SLOT(updateGL()));
   glWidget->drawXZPlaneAct = new QAction(tr("x-&z-Plane"), this);
   glWidget->drawXZPlaneAct->setCheckable(true);
   connect(glWidget->drawXZPlaneAct, SIGNAL(triggered()), glWidget, SLOT(updateGL()));
   glWidget->drawYZPlaneAct = new QAction(tr("&y-z-Plane"), this);
   glWidget->drawYZPlaneAct->setCheckable(true);
   connect(glWidget->drawYZPlaneAct, SIGNAL(triggered()), glWidget, SLOT(updateGL()));

   glWidget->bg = new QActionGroup(this);
   glWidget->bgBlack = new QAction(tr("&Black Background"), this);
   glWidget->bgBlack->setCheckable(true);
   glWidget->bg->addAction(glWidget->bgBlack);
   glWidget->bgWhite = new QAction(tr("&White Background"), this);
   glWidget->bgWhite->setCheckable(true);
   glWidget->bg->addAction(glWidget->bgWhite);
   glWidget->bgColor = new QAction(tr("&Colored Background"), this);
   glWidget->bgColor->setCheckable(true);
   glWidget->bg->addAction(glWidget->bgColor);
   glWidget->bgColor->setChecked(true);
   connect(glWidget->bg, SIGNAL(triggered(QAction*)), glWidget, SLOT(updateGL()));

   helpAct = new QAction(tr("&Help"), this);
   connect(helpAct, SIGNAL(triggered()), this, SLOT(help()));

   aboutAct = new QAction(tr("&About AMVis"), this);
   connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

   aboutQtAct = new QAction(tr("About &Qt"), this);
   connect(aboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
}

void MainWindow::createMenus()
{
   fileMenu = menuBar()->addMenu(tr("&File"));
   QMenu *winMenu=fileMenu->addMenu("&Window State");
     winMenu->addAction("&Load...", this, SLOT(loadWindowState()));
     winMenu->addAction("&Save...", this, SLOT(saveWindowState()));
   QMenu *viewMenu=fileMenu->addMenu("&View Port State");
     viewMenu->addAction("&Load...", this, SLOT(loadViewPortState()));
     viewMenu->addAction("&Save...", this, SLOT(saveViewPortState()));
   fileMenu->addSeparator();
   fileMenu->addAction(cachePosFilesAct);
   fileMenu->addSeparator();
   fileMenu->addAction(exportSinglePNGAct);
   fileMenu->addAction(exportPNGAct);
   fileMenu->addSeparator();
   fileMenu->addAction(exitAct);
   QMenu *editMenu=menuBar()->addMenu(tr("&Edit"));
   editMenu->addAction("Select Body [Alt+Click, Ctrl+Alt+Click]");
   editMenu->addAction("Move Camera With Body [Shift+Click]");
   viewMenu = menuBar()->addMenu(tr("&View"));
   viewMenu->addAction("&Top-View", glWidget, SLOT(look_from_top()), QKeySequence::fromString("Ctrl+T"));
   viewMenu->addAction("&Bottom-View", glWidget, SLOT(look_from_bottom()), QKeySequence::fromString("Ctrl+B"));
   viewMenu->addAction("Fr&ont-View", glWidget, SLOT(look_from_front()), QKeySequence::fromString("Ctrl+O"));
   viewMenu->addAction("Bac&k-View", glWidget, SLOT(look_from_back()), QKeySequence::fromString("Ctrl+K"));
   viewMenu->addAction("&Left-View", glWidget, SLOT(look_from_left()), QKeySequence::fromString("Ctrl+L"));
   viewMenu->addAction("&Right-View", glWidget, SLOT(look_from_right()), QKeySequence::fromString("Ctrl+R"));
   viewMenu->addSeparator();
   viewMenu->addAction(glWidget->drawWorldCosAct);
   viewMenu->addAction(glWidget->drawXYPlaneAct);
   viewMenu->addAction(glWidget->drawXZPlaneAct);
   viewMenu->addAction(glWidget->drawYZPlaneAct);
   viewMenu->addSeparator();
   viewMenu->addActions(glWidget->bg->actions());
   QMenu *dockMenu = menuBar()->addMenu(tr("&Docks"));
   dockMenu->addAction(LeftInfoListDock->toggleViewAction());
   dockMenu->addAction(ObjectPropertyBox->toggleViewAction());
   dockMenu->addAction(AnimationDock->toggleViewAction());
   helpMenu = menuBar()->addMenu(tr("&Help"));
   helpMenu->addAction(helpAct);
   helpMenu->addSeparator();
   helpMenu->addAction(aboutAct);
   helpMenu->addAction(aboutQtAct);
}


void MainWindow::set_window_title ( QString title)
{
   setWindowTitle(title);
};

void MainWindow::cachePosFiles()
{
  vector<CBody*>::iterator i;
  for(i=glWidget->NewGLContainer.OBjectVectorContainer.begin();
      i!=glWidget->NewGLContainer.OBjectVectorContainer.end(); i++)
    (*i)->cachePosFile();
};

void MainWindow::loadWindowState() {
  QString filename=QFileDialog::getOpenFileName(this, "Load Window State", QString(), "*.wst");

  QFile stateFile(filename);
  stateFile.open(QIODevice::ReadOnly);
  QByteArray state=stateFile.readAll();
  stateFile.close();

  restoreState(state, 0);
}

void MainWindow::saveWindowState() {
  QString filename=QFileDialog::getSaveFileName(this, "Save Window State", QString(), "*.wst");
  if(!filename.endsWith(".wst",Qt::CaseInsensitive))
    filename=filename+".wst";

  QByteArray state=saveState(0);

  QFile stateFile(filename, 0);
  stateFile.open(QIODevice::WriteOnly);
  stateFile.write(state);
  stateFile.close();
}

void MainWindow::loadViewPortState() {
  QString filename=QFileDialog::getOpenFileName(this, "Load View Port State", QString(), "*.vpt");
  if(!filename.endsWith(".vpt",Qt::CaseInsensitive))
	filename=filename+".vpt";
  loadViewPortStateFile(filename);
}

void MainWindow::loadViewPortStateFile(QString filename) {
  QFile stateFile(filename, 0);
  stateFile.open(QIODevice::ReadOnly);
  if(!stateFile.isOpen()) return;
  QTextStream in(&stateFile);

  QString dummy = in.readLine(); // "# GL_MODELVIEW_MATRIX"
  GLdouble state[16];
  for(int i=0;i<16;i++)
	in >> state[i];

  dummy = in.readLine(); // to end of line
  dummy = in.readLine(); // "# world KOS"
  int worldCOS;
  in >> worldCOS;
  glWidget->drawWorldCosAct->setChecked(worldCOS);

  dummy = in.readLine(); // to end of line
  dummy = in.readLine(); // "# planes"
  int plane[3];
  in >> plane[0] >> plane[1] >> plane[2];
  glWidget->drawXYPlaneAct->setChecked(plane[0]);
  glWidget->drawXZPlaneAct->setChecked(plane[1]);
  glWidget->drawYZPlaneAct->setChecked(plane[2]);

  dummy = in.readLine(); // to end of line
  dummy = in.readLine(); // "# background color"
  int bgColorBool=0;
  int bgColorIndex=-1;

  while(!bgColorBool && bgColorIndex<3-1) {
	in >> bgColorBool; 
	bgColorIndex++;
  }

  QList<QAction*> bgList = glWidget->bg->actions();
  bgList[bgColorIndex]->setChecked(true);

  glWidget->set_view_state(state);
 
  stateFile.close();
}

void MainWindow::saveViewPortState() {
// get data from GLWidget
  GLdouble state[16];
  glWidget->get_view_state(state);

// define outputfile
  QString filename=QFileDialog::getSaveFileName(this, "Save View Port State", QString(), "*.vpt");
  if(!filename.endsWith(".vpt",Qt::CaseInsensitive))
	filename=filename+".vpt";
  QFile stateFile(filename, 0);
  stateFile.open(QIODevice::WriteOnly);
  QTextStream out(&stateFile);

  out << "# GL_MODELVIEW_MATRIX";
  for(int i=0;i<16;i++) {
	if(!( i   %4)) out << endl;
	out << state[i+0];
	if( ((i+1)%4)) out << "\t";
  }

  out << endl << "# world KOS" << endl;
  out << glWidget->drawWorldCosAct->isChecked() << endl;

  out << "# planes" << endl;
  out << glWidget->drawXYPlaneAct->isChecked() << "\t" ;
  out << glWidget->drawXZPlaneAct->isChecked() << "\t" ;
  out << glWidget->drawYZPlaneAct->isChecked() << endl;

  out << "# background color" << endl;
  QList<QAction*> bgList = glWidget->bg->actions();
  for(int i=0;i<bgList.size();i++)
	out << bgList[i]->isChecked() << "\t" ;
  out << endl;

// close outputfile
  stateFile.close();
}

