/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <GL/gl.h>
#include "objobject.h" // class's header file

ObjObject::ObjObject(char* body_name, char * pos_file): CRigidBody( body_name,  pos_file)
{
   int result;
   strcpy(body_file_name,body_name);
   strcpy(posfile_name , pos_file);
   objfilename = new char [charlength];
   parsenextdataline(datafile);
   result = GetCharacterLine(objfilename, 128, datafile);

   int dummy;

   parsenextdataline(datafile);
   fscanf(datafile, "%d", &dummy);
   usetex=(bool)dummy;

   parsenextdataline(datafile);
   fscanf(datafile, "%d", &dummy);
   usemat=(bool)dummy;

   parsenextdataline(datafile);
   fscanf(datafile, "%d", &normalFlag);

   parsenextdataline(datafile);
   fscanf(datafile, "%f", &epsVertex);

   parsenextdataline(datafile);
   fscanf(datafile, "%f", &epsNormal);

   parsenextdataline(datafile);
   fscanf(datafile, "%f", &epsAngle);

   outline=1;
   parsenextdataline(datafile);
   int err=fscanf(datafile, "%d", &outline);
   if(err<1) printf("WARNING!!! This format of 'ObjObject'-body-file is deprecated!\n"
                    "           See menu 'Help/Help' for new format.\n");
}

ObjObject::ObjObject(FILE *bodyfile): CRigidBody(bodyfile)
{
   int result;
   objfilename = new char [charlength];
   parsenextdataline(datafile);
   result = GetCharacterLine(objfilename, 128, datafile);

   int dummy;

   parsenextdataline(datafile);
   fscanf(datafile, "%d", &dummy);
   usetex=(bool)dummy;

   parsenextdataline(datafile);
   fscanf(datafile, "%d", &dummy);
   usemat=(bool)dummy;

   parsenextdataline(datafile);
   fscanf(datafile, "%d", &normalFlag);

   parsenextdataline(datafile);
   fscanf(datafile, "%f", &epsVertex);

   parsenextdataline(datafile);
   fscanf(datafile, "%f", &epsNormal);

   parsenextdataline(datafile);
   fscanf(datafile, "%f", &epsAngle);

   outline=1;
   parsenextdataline(datafile);
   int err=fscanf(datafile, "%d", &outline);
   if(err<1) printf("WARNING!!! This format of 'ObjObject'-body-file is deprecated!\n"
                    "           See menu 'Help/Help' for new format.\n");
}


// class destructor
ObjObject::~ObjObject()
{
}

void ObjObject::init()
{
   CRigidBody::init();
   
   if(usetex)
     wfEnable(WF_USE_TEXTURES);
   else
     wfDisable(WF_USE_TEXTURES);
   if(usemat)
     wfEnable(WF_USE_MATERIALS);
   else
     wfDisable(WF_USE_MATERIALS);
   printf("AMVis: Reading Obj-file '%s'\n", objfilename);
   model=wfReadObject(objfilename);
   if(epsVertex>0) {
     printf("AMVis: Combine vertices\n");
     wfCombineVertices(model, epsVertex);
   }
   // normalFlag==0 use normals from obj-file
   if(normalFlag==1) {
     printf("AMVis: Compute flat normals\n");
     wfComputeNormals(model,false,false);
   }
   else if(normalFlag==2) {
     printf("AMVis: Compute smooth normals\n");
     wfComputeNormals(model,true,false);
   }
   else if(normalFlag==3) {
     printf("AMVis: Compute real normals\n");
     wfComputeNormals2(model, epsAngle);
   }
   if(epsNormal>0 && normalFlag==0) {
     printf("AMVis: Combine normals\n");
     wfCombineNormals(model, epsNormal);
   }
   wfInitObject(model);
   contourList=0;
   if(outline==1) {
     printf("AMVis: Compute outline\n");
     contourList=wfGenOutline(model);
//FILE*f=fopen("out.obj","w");
//wfWriteObject(f,contourList);
//fclose(f);
   }
   if(outline==2) {
     char *outlineFile=new char[strlen(objfilename)+80];
     strcpy(outlineFile, objfilename);
     strcpy(outlineFile+strlen(objfilename)-4,".outline.obj");
     printf("AMVis: Reading outline from Obj-file '%s'\n", outlineFile);
     contourList=wfReadObject(outlineFile);
   }

   float c0[3], c1[3];
   wfGetBoundingBox(model,c0,c1);
   updateminmax(c0);
   updateminmax(c1);
}

void ObjObject::draw_contour()
{
  glEnable(GL_POLYGON_OFFSET_LINE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  glEnable(GL_COLOR_MATERIAL);
  glLineWidth(1.0);
  glPolygonOffset(OFFSETFAC1,OFFSETUNIT1);
  glDisable(GL_LIGHTING); 
  glEnable(GL_LINE_SMOOTH);
  glColor3f(0.0, 0.0, 0.0);
  if(contourList)
    wfDrawObject(contourList);
  glPolygonOffset(0.0,0.0);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_POLYGON_OFFSET_LINE);
}

void ObjObject::draw_pure()
{
   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   glDisable(GL_LIGHTING);
   glEnable(GL_COLOR_MATERIAL);
   glColor3fv(pick_color_value);
   glPushMatrix();
   glTranslated(position_x, position_y, position_z);
   glMultMatrixf(object_rotation_matrix);
   glTranslated(position_x_initial, position_y_initial, position_z_initial);
   glMultMatrixf(object_initial_matrix);
   glScaled(scale_factor, scale_factor, scale_factor);

   //glCallList(Displ_List);
   wfDisable(WF_USE_TEXTURES);
   wfDisable(WF_USE_MATERIALS);
   wfDrawObject(model);

   glPopMatrix();
   glEnable(GL_LIGHTING);
   glDisable(GL_COLOR_MATERIAL);
}

// Can not call a Display-List because wfDrawObject calls glNewList
void ObjObject::draw()
{
   glLineWidth(1.0);
   glPointSize(2.0);
   glEnable(GL_LINE_SMOOTH);
   glEnable(GL_POINT_SMOOTH);
   if (mode ==0) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   if (mode ==1) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
   if (mode ==2) glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);

   glDisable(GL_COLOR_MATERIAL);
   glEnable(GL_LIGHTING);
   updateRotationMatrix();
   createInitialRotationMatrix();
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glPolygonOffset(OFFSETFAC0,OFFSETUNIT0);
   glEnable(GL_POLYGON_OFFSET_FILL);
   glEnable(GL_POLYGON_OFFSET_LINE);
   glEnable(GL_POLYGON_OFFSET_POINT);
   glEnable(GL_BLEND);
   glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

   //global translation and rotation
   glTranslated(position_x, position_y, position_z);
   glMultMatrixf(object_rotation_matrix);

   if (local_cos_flag==true) // for R system
   {
      float abs=fmax(fmax(x_max,y_max),z_max);

      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glLineWidth(3.0);
      // X DIR red
      glColor3ub(90, 0, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(abs* 1.5, 0.0, 0.0);
      glEnd();
      // Y DIR green
      glColor3ub(0, 90, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, abs* 1.5, 0.0);
      glEnd();
      // Z DOR blue
      glColor3ub(0, 0, 90);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, 0.0, abs* 1.5);
      glEnd();
      glEnable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
      glLineWidth(1.0);
   };

   translateColorvalueToRGBA(colorvalue, &color[0], &color[1], &color[2], &color[3]);
   glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
   GLfloat mat_specular[] = {1.0,1.0,1.0, 1.0};
   glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
   float em[4]={0,0,0,0};
   glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);

   //initial translation and roation
   glTranslated(position_x_initial, position_y_initial, position_z_initial);
   glMultMatrixf(object_initial_matrix);
   glScaled(scale_factor, scale_factor, scale_factor);

   //glCallList(Displ_List);
   if(usetex)
     wfEnable(WF_USE_TEXTURES);
   else
     wfDisable(WF_USE_TEXTURES);

   if(usemat)
     wfEnable(WF_USE_MATERIALS);
   else
     wfDisable(WF_USE_MATERIALS);

   wfDrawObject(model);

   glPolygonOffset(0.0,0.0);
   if (contour_flag==true)
   {
      draw_contour();
   };

   if (selected_flag==true)
   {
      glPolygonOffset(OFFSETFAC1,OFFSETUNIT1);
      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glColor3f(0.0, 1.0, 0.0);
      glLineWidth(4.0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_min, y_max, z_min);
      glVertex3f(x_min, y_max, z_max);
      glVertex3f(x_min, y_min, z_max);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_max, y_min, z_min);
      glVertex3f(x_max, y_max, z_min);
      glVertex3f(x_max, y_max, z_max);
      glVertex3f(x_max, y_min, z_max);
      glVertex3f(x_max, y_min, z_min);
      glEnd();
      glBegin(GL_LINES);
      glVertex3f(x_min, y_min, z_min);
      glVertex3f(x_max, y_min, z_min);
      glVertex3f(x_min, y_max, z_min);
      glVertex3f(x_max, y_max, z_min);
      glVertex3f(x_min, y_max, z_max);
      glVertex3f(x_max, y_max, z_max);
      glVertex3f(x_min, y_min, z_max);
      glVertex3f(x_max, y_min, z_max);
      glEnd();
      glEnable(GL_LIGHTING);
      glPolygonOffset(0.0,0.0);
      glEnable(GL_POLYGON_OFFSET_LINE);
      glDisable(GL_COLOR_MATERIAL);
   };


   if (local_cos_flag==true) // for L system
   {
      float abs=fmax(fmax(x_max,y_max),z_max);

      glEnable(GL_COLOR_MATERIAL);
      glDisable(GL_LIGHTING);
      glEnable(GL_LINE_SMOOTH);
      glLineWidth(3.0);
      // X DIR red
      glColor3ub(255, 0, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(abs* 1.5, 0.0, 0.0);
      glEnd();
      // Y DIR green
      glColor3ub(0, 255, 0);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, abs* 1.5, 0.0);
      glEnd();
      // Z DOR blue
      glColor3ub(0, 0, 255);
      glBegin(GL_LINE_STRIP);
      glVertex3f(0.0, 0.0, 0.0);
      glVertex3f(0.0, 0.0, abs* 1.5);
      glEnd();
      glEnable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
   };

   glDisable(GL_POLYGON_OFFSET_FILL);
   glDisable(GL_POLYGON_OFFSET_LINE);
   glDisable(GL_POLYGON_OFFSET_POINT);
   glPolygonOffset(0.0,0.0);
   glPopMatrix();

  if(path_flag) {
    long drawpathto=curdataset;
    if(path.size()<=(unsigned long)drawpathto) {
      for(int i=path.size(); i<=drawpathto; i++) {
        update(i);
        Point3D *p=new Point3D;
        p->x=position_x;
        p->y=position_y;
        p->z=position_z;
        path.push_back(p);
      }
    }
    drawPath(0, 1, 0);
  }
}
