/*
    AMVis - 3D Multibody Visualisation Program.
    Copyright (C) 2006 Institute of Applied Mechanics,
                       Technische Universitaet Muenchen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <GL/gl.h>
#include "quad.h" // class's header file

using namespace std;

// class constructor
Quad::Quad(char* body_name, char * pos_file): ElasticBody( body_name,  pos_file)
{
   dofs = 12;
   q = new float[dofs];
   binaryrecordsize=sizeof(float)*(dofs + 1); // Zeit nicht vergessen
}

Quad::~Quad()
{
}

void Quad::init()
{
    ElasticBody::init();
    for(int i=0;i<dofs;i++) q[i] = 0;
    q[ 3] = 1;
    q[ 6] = 1;
    q[ 7] = 1;
    q[10] = 1;
}

void Quad::draw()
{
    ElasticBody::draw();

   // Farben
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial (GL_FRONT_AND_BACK, GL_DIFFUSE);
    GLfloat color[] = {1.2,2.2,1.2,1.8};
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
    GLfloat mat_specular[] = {1.25,1.9,1.25, 1.1};
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
    GLfloat color_shininess = 12;
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &color_shininess);
    float em[4]={0,0,0,0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, em);


    drawBodyFast(); // this calls "virtual void drawBody()" in a very fast way if possible!

    if(contour_flag==true) {
	// draw contour
	glEnable(GL_POLYGON_OFFSET_LINE);
	glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	glLineWidth(1.0);
	glPolygonOffset(-1.0,0.0);
	glDisable(GL_LIGHTING);
	
	drawBodyOutline();
    }
    
    if(selected_flag==true) {
	// draw something
    }
    
    if(local_cos_flag==true) {
	// draw something
    }
}

void Quad::drawBody() {
    static const int    divisions = 16;
    static const double     Delta = 1./divisions;
    
    glBegin( GL_QUAD_STRIP );
    glNormal3dv(normal(0.5*Delta,0.5));	
    for (int x=0;x<=divisions;x++) {
	if(1<x)
	    glNormal3dv(normal((x-0.5)*Delta,0.5));	
	glVertex3dv(position(x*Delta,0.0));
	glVertex3dv(position(x*Delta,1.0));
    }
    glEnd();
}

void Quad::drawBodyOutline() {
  glEnable(GL_LINE_SMOOTH);
  glColor3f(0.0, 0.0, 0.0);

  glBegin( GL_LINES );
  glVertex3dv(position(0,0));    glVertex3dv(position(1,0));
  glVertex3dv(position(1,0));    glVertex3dv(position(1,1));
  glVertex3dv(position(1,1));    glVertex3dv(position(0,1));
  glVertex3dv(position(0,1));    glVertex3dv(position(0,0));
  glEnd();

  static const int    divisions = 4;
  static const double     Delta = 1./divisions;
  glColor3f(0.4, 0.6, 0.4);
  for (int x=1;x<divisions;x++) {
      glBegin( GL_LINES );
      glVertex3dv(position(x*Delta,0)); glVertex3dv(position(x*Delta,1));
      glVertex3dv(position(0,x*Delta)); glVertex3dv(position(1,x*Delta));
      glEnd();
  }
}


GLdouble* Quad::position(const double &x, const double &y) {
    GLdouble *X = new GLdouble[3];
    X[0] = (1-x)*(1-y)*q[0] + (  x)*(1-y)*q[3] + (  x)*(  y)*q[6] + (1-x)*(  y)*q[ 9];
    X[1] = (1-x)*(1-y)*q[1] + (  x)*(1-y)*q[4] + (  x)*(  y)*q[7] + (1-x)*(  y)*q[10];
    X[2] = (1-x)*(1-y)*q[2] + (  x)*(1-y)*q[5] + (  x)*(  y)*q[8] + (1-x)*(  y)*q[11];

    return X;
}

GLdouble* Quad::normal(const double &x, const double &y) {
    GLdouble *X = new GLdouble[3];
    const double &x1 = q[0]; const double &y1 = q[ 1]; const double &z1 = q[ 2];
    const double &x2 = q[3]; const double &y2 = q[ 4]; const double &z2 = q[ 5];
    const double &x3 = q[6]; const double &y3 = q[ 7]; const double &z3 = q[ 8];
    const double &x4 = q[9]; const double &y4 = q[10]; const double &z4 = q[11];

    X[0] = -( -y2*z1 + y4*z1 + y1*z2 - y4*z2 - y1*z4 + y2*z4 + 
	x*(y3*z1 - y4*z1 - y3*z2 + y4*z2 - y1*z3 + y2*z3 + y1*z4 - y2*z4) + 
	y*(y2*z1 - y3*z1 - y1*z2 + y4*z2 + y1*z3 - y4*z3 - y2*z4 + y3*z4) );

    X[1] = -( x2*z1 - x4*z1 - x1*z2 + x4*z2 + x1*z4 - x2*z4 + 
	y*(-x2*z1 + x3*z1 + x1*z2 - x4*z2 - x1*z3 + x4*z3 + x2*z4 - x3*z4) + 
	x*(x4*(z1 - z2) + x3*(-z1 + z2) + x1*(z3 - z4) + x2*(-z3 + z4)) );

    X[2] = -( -x2*y1 + x4*y1 + x1*y2 - x4*y2 - x1*y4 + x2*y4 + 
	x*(x3*y1 - x4*y1 - x3*y2 + x4*y2 - x1*y3 + x2*y3 + x1*y4 - x2*y4) + 
	y*(x2*y1 - x3*y1 - x1*y2 + x4*y2 + x1*y3 - x4*y3 - x2*y4 + x3*y4) );

    double nrm = sqrt( X[0]*X[0] + X[1]*X[1] + X[2]*X[2] );
    X[0] /= nrm;    X[1] /= nrm;    X[2] /= nrm; 
    
    return X;
}

void Quad::draw_pure() {
    ElasticBody::draw_pure();

    glEnable(GL_SMOOTH);


    GLdouble p[4][3];
    for (int j=0;j<4;j++)
	for (int i=0;i<3;i++) p[j][i] = q[3*j+i];
    glBegin( GL_QUADS );
    glVertex3dv(p[0]);  glVertex3dv(p[1]);  glVertex3dv(p[2]);  glVertex3dv(p[3]);
    glEnd();
}
